--      VECTORS.PAS     -- vector engine by FAC
--
--      this unit defines the TVector and TVertex classes,
--      as well as some functions to work with vectors and
--      the SinT and CosT precalculated tables.
--
--      this unit is 100% in pascal and it's slow, maybe i'll
--      optimize it for the FPU someday.
--
--      All class members and methods are declared as public


unit Vectors;


interface

{ Vectores -- TVector class }
type PTVector = ^TVector;  { pointer to a vector object }
     TVector = object      { vector class definition    }
             public
                   x, y, z : single;    { vector components }

                   constructor Init(nx, ny, nz : single);    { initialize   }
                   procedure SetP(nx, ny, nz : single);      { set position }
                   procedure Normalize;                      { normalize    }
                   function Magnitude : single;              { magnitude    }
                   procedure Traslate(tx, ty, tz : single);  { traslation   }
                   procedure Rotate(ax, ay, az : integer);   { rotation     }
                   procedure Scale(sx, sy, sz : single);     { scaling      }
             end;

{ Procedimientos para trabajar con vectores }
{ some common vector operations }

function DotProduct(a, b : TVector) : single;
procedure CrossProduct(a, b : TVector; var r : TVector);
procedure AddVectors(a, b : TVector; var r : TVector);
procedure SubVectors(a, b : TVector; var r : TVector);
procedure MulVector(var v : TVector; k : single);
procedure DivVector(var v : TVector; k : single);
procedure CopyVector(source : TVector; var dest : TVector);


{ TVertex class }
type PTVertex = ^TVertex;       { pointer to a vertex object }
     TVertex = object(TVector)  { vertex class definition }
             public
                   normal : TVector;    { vertex normal }
                   x2d, y2d : integer;  { 2D projection }
                   u, v : single;       { UV (texture) coordinates      }
                   c : byte;            { intensity of light (color)    }

                   constructor Init(nx, ny, nz : single);   { initialize }
                   procedure Calc2D(xc, yc : integer);      { calc projection }
                   procedure VRotate(ax, ay, az : integer); { rotate }
                   procedure FakeNormal; { calculates a fake normal  }
             end;


{ trigonometric tables }
var SinT, CosT : array[0..359] of single;


implementation

{ vector class implementation }

{ Init - constructor }
constructor TVector.Init(nx, ny, nz : single);
begin
     SetP(nx, ny, nz);
end;

{ SetP - fija las componentes del vector -- set vector components }
procedure TVector.SetP(nx, ny, nz : single);
begin
     x := nx;
     y := ny;
     z := nz;
end;

{ Normalize - normaliza el vector -- normalizes the vector }
procedure TVector.Normalize;
var mag : single;
begin
     mag := Magnitude;
     if mag = 0 then exit;
     x := x / mag;
     y := y / mag;
     z := z / mag;
end;

{ Magnitude - devuelve la magnitud del vector -- returns the vector magnitude }
function TVector.Magnitude : single;
begin
     Magnitude := sqrt(x * x + y * y + z * z);
end;

{ Traslate - traslada el vector -- traslates the vector }
procedure TVector.Traslate(tx, ty, tz : single);
begin
     x := x + tx;
     y := y + ty;
     z := z + tz;
end;

{ Rotate - rota el vector con respecto a cada eje. }
{ rotates the vector around each axis              }
procedure TVector.Rotate(ax, ay, az : integer);
var nx, ny, nz : single;
begin
     nx := x;   { new coordinates }
     ny := y;
     nz := z;

     { make the angle between 0 and 360 (for negative angles) }
     if ax < 0 then inc(ax, 360);
     if ay < 0 then inc(ay, 360);
     if az < 0 then inc(az, 360);

     { X-axis rotation }
     if ax <> 0 then
     begin
          ny := y * CosT[ax] - z * SinT[ax];
          nz := y * SinT[ax] + z * CosT[ax];
          y := ny;
          z := nz;
     end;

     { Y-axis rotation }
     if ay <> 0 then
     begin
          nx := x * CosT[ay] - z * SinT[ay];
          nz := x * SinT[ay] + z * CosT[ay];
          x := nx;
          z := nz;
     end;

     { Z-axis rotation }
     if az <> 0 then
     begin
          nx := x * CosT[az] - y * SinT[az];
          ny := x * SinT[az] + y * CosT[az];
          x := nx;
          y := ny;
     end;
end;


{ vector scaling procedure }
procedure TVector.Scale(sx, sy, sz : single);
begin
     x := x * sx;
     y := y * sy;
     z := z * sz;
end;



{ common vector operations }

{ DotProduct - producto punto }
function DotProduct(a, b : TVector) : single;
begin
     DotProduct := a.x * b.x + a.y * b.y + a.z * b.z;
end;

{ CrossProduct - producto cruz. resultado en el tercer parmetro }
procedure CrossProduct(a, b : TVector; var r : TVector);
begin
     r.x := a.y * b.z - a.z * b.y;
     r.y := b.x * a.z - a.x * b.z;
     r.z := a.x * b.y - a.y * b.x;
end;

{ AddVectors - suma vectores }
procedure AddVectors(a, b : TVector; var r : TVector);
begin
     r.x := a.x + b.x;
     r.y := a.y + b.y;
     r.z := a.z + b.z;
end;

{ SubVectors - resta vectores }
procedure SubVectors(a, b : TVector; var r : TVector);
begin
     r.x := a.x - b.x;
     r.y := a.y - b.y;
     r.z := a.z - b.z;
end;

{ MulVector - multiplica un vector por una constante }
procedure MulVector(var v : TVector; k : single);
begin
     v.x := v.x * k;
     v.y := v.y * k;
     v.z := v.z * k;
end;

{ DivVector - divide un vector entre una constante (sin comprobacin) }
procedure DivVector(var v: TVector; k : single);
begin
     v.x := v.x / k;
     v.y := v.y / k;
     v.z := v.z / k;
end;

{ CopyVector - hace el vector destino igual al origen                      }
{ this function is used because  dest := source  doesn't work with objects }
procedure CopyVector(source : TVector; var dest : TVector);
begin
     dest.x := source.x;
     dest.y := source.y;
     dest.z := source.z;
end;


{ TVertex implementation }

{ Init - constructor }
constructor TVertex.Init(nx, ny, nz : single);
begin
     SetP(nx, ny, nz);
     normal.Init(0, 0, 0);
     x2d := 0;
     y2d := 0;
     u := 0;
     v := 0;
     c := 0;
end;


{ Calc2D - calculates the 2D projection of the vertex                   }
{ (xc, yc) are the screen coordinates where the 3D origin is located    }
procedure TVertex.Calc2D(xc, yc : integer);
begin
     x2d := trunc(x / (1024 - z) * 512) + xc;
     y2d := trunc(y / (z - 1024) * 512) + yc;
end;

{ VRotate - rotates the vertex and it's normal }
procedure TVertex.VRotate(ax, ay, az : integer);
begin
     Rotate(ax, ay, az);
     normal.Rotate(ax, ay, az);
end;

{ FakeNormal - calculates a fake and inaccurate normal (not used in the demo)}
procedure TVertex.FakeNormal;
begin
     normal.x := x;
     normal.y := y;
     normal.z := z;
     normal.Normalize;
end;


{ this one here calculates the SinT and CosT tables }
procedure CalcTrigTables;
var ang : integer;
begin
     for ang := 0 to 359 do
     begin
          SinT[ang] := sin(ang * Pi / 180);
          CosT[ang] := cos(ang * Pi / 180);
     end;
end;

{ unit initialization }
begin
     CalcTrigTables;
end.
