/*  detect.c
 *  preliminary alpha-version
 *
 *  This file is part of the MMM, multi system multi format
 *  music handler, layer 2 (no hw,os dependencies).
 *
 *  No 'huge' memory is needed. So we use the own routines
 *  tmp_alloc(), tmp_free() similar to malloc(), free().
 *
 *  (do not use 'implicite' increments like a[i++]=b[i];
 *   VAX-C has a different behaviour than other C-compilers.)
 *
 *  Mar/94, Harald Zappe
 *          zappe@gaea.sietec.de
 */

#define _TEST_
#define ANSI

#include <stdio.h>
#include <io.h>
#include <string.h>

/* ------------------------------------------------------------ */
/* #include "mmmtypes.h" */

typedef unsigned char UCHAR;
typedef unsigned long ULONG;

typedef enum MUS_TYPE {
                        MUS_UNKNOWN   = 0x0000,
                        MUS_MOD_OLD15 = 0x1004, /* Karsten Obarki       */
                        MUS_MOD_PT4   = 0x1014, /* Mahoni & Kaktus      */
                        MUS_MOD_PT6   = 0x1016, /*                      */
                        MUS_MOD_PT8   = 0x1018, /*                      */

                        MUS_STS       = 0x1020, /* Scream Tracker       */
                        MUS_STM       = 0x1024, /* Scream Tracker       */
                        MUS_GTS       = 0x1030, /*                      */

                        MUS_MED       = 0x1044, /* ?                    */

                        MUS_669       = 0x1068, /* 669-Composer         */
                        MUS_Oktalyser = 0x1088, /*                      */
                        MUS_FORM_SMUS = 0x1198, /* IFF-style            */
                        MUS_FORM_TRKR = 0x11AF, /* IFF-style            */
                        MUS_FORM_MODL = 0x11BF, /* IFF-style            */
                        MUS_FORM_EMOD = 0x11CF, /* IFF-style            */
                        MUS_MIFF      = 0x11DF, /* IFF-style            */
                        MUS_TFMX      = 0x120F, /* Cris H"ulsbeck       */
                        MUS_S3M       = 0x121F, /* Future Crew          */
                        MUS_ULTRA     = 0x122F, /*                      */

                        MUS_ROL       = 0x2001, /* Adlib                */
                        MUS_CMF       = 0x2002, /* Creativ Music File   */
                        MUS_SID       = 0x2003, /* Commodore            */

                        MUS_MMD0      = 0x3054, /* OktaMED              */
                        MUS_MMD1      = 0x3058, /* OktaMED              */
                        MUS_MIDI      = 0x3000, /* midi                 */

                        OTHERS_TEXT   = 0x7F7F, /* ascii text (7 bit)   */
                        OTHERS_NONE   = 0x7F00, /* NULL-fp              */
                        OTHERS_EMPTY  = 0x7F01  /* 0 byte file          */
           } MUS_T;

/* #include "mmmmem.h" ? */


extern char *tmp_alloc (unsigned short len);
extern short tmp_free  (char *mem);

/* unsigned flag; (not needed yet) */

/* ------------------------------------------------------------ */
/*  Check whether the given string represents a valid DOS file
 *  name: "12345678.123"
 */

/* static */
int valid_dosfilename (char *name) {

  int   i = 0;
  UCHAR c;

  while ((i < 12) && (name[i] != 0)) {
    c = name[i];
    if (   (c < 0x20) || (c == '*')
        || (c == '<') || (c == '>')
        || (c == '?')
       ) {
      return 0; /* invalid */
    }
    i++;
  }

  if (i == 0) return (0); /* invalid (empty) */
  else        return (1); /* valid */

} /* valid_dosfilename */

/* ------------------------------------------------------------ */
/*
 *  Find out which type of <music file> the given file is. The
 *  difficulty here is that we may not be too strong, where we
 *  loose valid variants. On the other hand, if the tests are
 *  too lazy, foreign data files (eg. pictures) may be incorrect-
 *  ly detected as <music files>.
 *  Because some tests are not very sure, we allow a multiple
 *  choice answer, eg. MOD15 or MMD0, ROL or GTS.
 *  Some special types help to tell that it is NOT a valid
 *  <music file> (eg. text).
 */

MUS_T *detect_music_filetype (FILE *fp) {

  /* fp: pointer to already (binary) opened file */

  /*  Because renaming a file is quite simple, we don't believe
   *  any fairytails and only believe what we see.
   *  Here we can NOT know the name of the file, and we really
   *  do not want. (HZ)
   *  On a MAC do NOT use the finder fork. Here no assumptions
   *  are made about the distribution to resource and data fork.
   *  On a VAX the file should have stream-lf format.
   */

# define HEADER_SIZE 2048
# define MAX_ASSMPT    16
  UCHAR        *header_buf;             /* for the 1st bytes of the file */
  int           read_cnt, i, j;
  int           e1, e2, e3;
  int           discard;                /* through away previous assumpt.*/
  ULONG         filesize;
  static MUS_T  assumption[MAX_ASSMPT]; /* list of possible types        */

  /* ---------------------------------------------------------- */

  /* until now we don't know anything */
  for (i=0; i<MAX_ASSMPT; i++) {
    assumption[i] = MUS_UNKNOWN;
  }

  if (fp == NULL) {
    /* no file given here */
    /* (eg. tried to open read-only file for writing) */
    assumption[0] = OTHERS_NONE;
    return (&assumption[0]);
  }

  /* get a bit of memory where to store the header */
  header_buf = (UCHAR *) tmp_alloc (HEADER_SIZE);
  if (header_buf == NULL) {
    /* no mem, cannot determine */
    /* should never occure !!!  */
    return (&assumption[0]);
  }

  /* get the first bytes of the questionable file */
  read_cnt = fread (header_buf, 1, HEADER_SIZE, fp);

  /* filesize = filelength (fileno (fp)); (MSC) */
  filesize = lseek (fileno(fp), 0, SEEK_END);

  if (read_cnt == 0) {
    /* nothing read, empty file */
    assumption[0] = OTHERS_EMPTY;
    (void) tmp_free ((char *)header_buf);
    return (&assumption[0]);
  }

  /* go back to the start, as if nothing has happened */
  rewind (fp); /* err = fseek (fp, 0L, SEEK_SET); */

  /* flag = 0; */
  i    = 0;

  if (read_cnt < HEADER_SIZE) {
    for (j=read_cnt; j<HEADER_SIZE; j++) {
      header_buf[j] = 0xFF;
    }
    /* file is too short for some tests */
    goto Others; /* 'goto' is a bit brutal, but... */
  }

  /* ---------------------------------------------------------- */
  /* Now we can analyze the first bytes of the header.       */
  /* Check the magics (positive check) and some other stuff  */

  /*  The old 15-instr., PT-Mods and STMs can have any text at
   *  the very beginning, where the other formats have their magic.
   *  So we test all locations.
   *  Those formats which are not so sure should be checked last.
   */
  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* First we check location 0, where a proper magic belongs to */

  /* MED: ? endian */
  if      (strncmp (&header_buf[0x0000], "MED\004", 4) == 0) {
    assumption[i] = MUS_MED; i++;
  }

  /* 669: big endian */
  else if (strncmp (&header_buf[0x0000], "if",      2) == 0) {
    if (   (header_buf[0x006E] <= 64)
        && (header_buf[0x006F] <= 128)
       ) {
      assumption[i] = MUS_669; i++;
    }
  }

  /* OktaMed: little endian */
  else if (strncmp (&header_buf[0x0000], "MMD",     3) == 0) {
    j = discard = 0;
    while (j < 17) {
      if (header_buf[778+j]-1 > 63) { /* trkvol */
        discard = 1;
        break;
      }
      j++;
    }
    if (!discard) {
      if (header_buf[0x0003] == '0') {
        assumption[i] = MUS_MMD0; i++;
      }
      else if (header_buf[0x0003] == '1') {
        assumption[i] = MUS_MMD1; i++;
      }
    }
  }

  /* Oktalyzer: little endian */
  else if (strncmp (&header_buf[0x0000], "OKTASONGCMOD", 12) == 0) {
    if (strncmp (&header_buf[0x0018], "SAMP", 4) == 0) {
      assumption[i] = MUS_Oktalyser; i++;
    }
  }

  /* IFF-styles: little endian */
  else if (strncmp (&header_buf[0x0000], "FORM",    4) == 0) {
    if (strncmp (&header_buf[0x0008], "SMUS", 4) == 0) {
      assumption[i] = MUS_FORM_SMUS; i++;
    }
    if (strncmp (&header_buf[0x0008], "TRKR", 4) == 0) {
      assumption[i] = MUS_FORM_TRKR; i++;
    }
    if (strncmp (&header_buf[0x0008], "MODL", 4) == 0) {
      assumption[i] = MUS_FORM_MODL; i++;
    }
    if (strncmp (&header_buf[0x0008], "EMOD", 4) == 0) {
      assumption[i] = MUS_FORM_EMOD; i++;
    }
  }
  else if (strncmp (&header_buf[0x0000], "MIFF",    4) == 0) {
    assumption[i] = MUS_MIFF; i++;
  }

  /* TFMX 2.0: little endian */
  else if (strncmp (&header_buf[0x0000], "TFMX-SONG ",   10) == 0) {
    assumption[i] = MUS_TFMX; i++;
  }

  else if (strncmp (&header_buf[0x0000], "MAS_UTrack_V", 12) == 0) {
    assumption[i] = MUS_ULTRA; i++;
  }

  /* CTMF: big endian */
  else if (strncmp (&header_buf[0x0000], "CTMF",    4) == 0) {
    if (   (header_buf[0x0004] < 16) /* file vers */
        && (header_buf[0x0005] < 16)
       ) {
      assumption[i] = MUS_CMF; i++;
    }
  }
  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* MOD15: little endian */

  if (   (header_buf[  44 ] == 0)
      && (header_buf[ 470 ] <= 63)   /* #patt */
     ) {
    ULONG         slen;
    ULONG         slen_sum;

    j = discard = 0;
    slen_sum = 0;

    while (j < 14) {

      if (   (header_buf[45+j*30] > 0x40)
          || (header_buf[44+j*30] !=  0)   /* vol */
         ){
        discard = 1;
        break;
      }
      slen = ((ULONG)header_buf[42+j*30] << 9)
           + ((ULONG)header_buf[43+j*30] << 1); /* sample len */
      slen_sum += slen;

#if 1
      if (slen_sum > filesize) {
        discard = 1;
        break;
      }
#endif
      j++;
    }
    if (!discard) {
      assumption[i] = MUS_MOD_OLD15; i++;
    }
  }
  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* MOD-PT: little endian */

  if (   (header_buf[ 950 ]-1 <  128 ) /* song len */
      && (header_buf[ 951 ]   <= 128 )
     ) {
    j = discard = 0;
    while (j < 128) {
      if (header_buf[952+j] > 127 /*63*/) {    /* song pos */
        discard = 1;
        break;
      }
      j++;
    }
    j = 0;
    while (j < 30) {
      if (header_buf[45+j*30] > 0x40) { /* vol */
        discard |= 1;
        break;
      }
      j++;
    }
    if (!discard) {
      if (   (strncmp (&header_buf[0x0438], "M.K.", 4) == 0)
          || (strncmp (&header_buf[0x0438], "M&K!", 4) == 0)
          || (strncmp (&header_buf[0x0438], "M!K!", 4) == 0)
          || (strncmp (&header_buf[0x0438], "FLT4", 4) == 0)
         ) {
        assumption[i] = MUS_MOD_PT4; i++;
      }
      else if (   (strncmp (&header_buf[0x0438], "CHN6", 4) == 0) /*not used*/
               || (strncmp (&header_buf[0x0438], "6CHN", 4) == 0)
              ) {
        assumption[i] = MUS_MOD_PT6; i++;
      }
      else if (   (strncmp (&header_buf[0x0438], "FLT8", 4) == 0)
               || (strncmp (&header_buf[0x0438], "CHN8", 4) == 0) /*not used*/
               || (strncmp (&header_buf[0x0438], "8CHN", 4) == 0)
              ) {
        assumption[i] = MUS_MOD_PT8; i++;
      }
      if (i > 1) {
        if (assumption[i-2] == MUS_MOD_OLD15) {
          i--;
          /* PT superseeds MOD15 */
          assumption[i-1] = assumption[i];
          assumption[i]   = MUS_UNKNOWN;
        }
      }
    }
  }
  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* STS/STM: big endian */

  if (   (header_buf[0x001C] == 0x1A)
      && (header_buf[0x003C] == 0)
     ) {
    j = discard = 0;
    while (j < 28) {
      if (header_buf[j] > 0x7F) {
        discard = 1;
        break;
      }
      j++;
    }
    if (!discard) {
      if      (header_buf[0x001D] == 1) { /* file type */
        assumption[i] = MUS_STS; i++;
      }
      else if (header_buf[0x001D] == 2) {
        assumption[i] = MUS_STM; i++;
      }
    }
  }

  /* S3M: */
  if (strncmp (&header_buf[0x002C], "SCRM",    4) == 0) {
    if (   (header_buf[0x001C] == 0x1A)
        && (header_buf[0x001D] == 0x10)
       ) {
      assumption[i] = MUS_S3M; i++;
    }
  }

  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* Some <music> files can be smaller than 2KB */
Others:

  /* Midi: little endian */
  if (strncmp (&header_buf[0x0000], "MThd",    4) == 0) {
    if (   (header_buf[0x0004] == 0)
        && (header_buf[0x0005] == 0) /* hdr len high */
       ) {
      /* here high part = 0, get low part */
      j = (header_buf[6]<<8)+header_buf[7];
      if (j < HEADER_SIZE-8) {
        /* look at next chunk */
        if (strncmp (&header_buf[8+j], "MTrk", 4) == 0) {
          assumption[i] = MUS_MIDI; i++;
        }
      }
    }
  }

  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* Sid: mixed Endian */
  e1 = header_buf[2] + (0x100*header_buf[3]) + (8-2);
  e2 = header_buf[4] + (0x100*header_buf[5]) + e1;
  e3 = header_buf[6] + (0x100*header_buf[7]) + e2;
  discard = 0;

  if (read_cnt <= HEADER_SIZE) {
    if (   (e1 >= read_cnt)
        || (e2 >= read_cnt)
        || (e3 >= read_cnt)
       ) {
      discard = 1;
    }
  }
  else {
    if (   (e1 >= filesize)
        || (e2 >= filesize)
        || (e3 >= filesize)
       ) {
      discard = 1;
    }
  }

  if ( (e1 < read_cnt-1) && !discard ) {
    if (   (header_buf[e1]   != 0x01)
        || (header_buf[e1+1] != 0x4F)
        || (e1 <= 6)
       ) {
      discard = 1;
    }
  }
  if ( (e2 < read_cnt-1) && !discard ) {
    if (   (header_buf[e2]   != 0x01)
        || (header_buf[e2+1] != 0x4F)
        || (e2 <= 6)
       ) {
      discard = 1;
    }
  }
  if ( (e3 < read_cnt-1) && !discard ) {
    if (   (header_buf[e3]   != 0x01)
        || (header_buf[e3+1] != 0x4F)
        || (e3 <= 6)
       ) {
      discard = 1;
    }
  }
  if (!discard) {
    assumption[i] = MUS_SID; i++;
  }

  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* ROL: big endian */
  if (   (header_buf[0x0000] <= 0x0F) /* ver maj, low  */
      && (header_buf[0x0001] == 0)    /* ver maj, high */
      && (header_buf[0x0002] <= 0x0F) /* ver min, low  */
      && (header_buf[0x0003] == 0)    /* ver min, high */
      && (header_buf[0x0035] <= 1)    /* perc/melod    */
     ) {
    /* not very sure, but let's assume ROL */
    assumption[i] = MUS_ROL; i++;
    /* should be checked a bit better (if possible) */
  }
  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* GTS: big endian */
  if (   (header_buf[0] < 0x20)
      && (header_buf[0] >    0)
     ) {
    j = discard = 0;
    while (j < (int)header_buf[0]) {     /* nr samples */
      if (!valid_dosfilename ((char *)&header_buf[1+(j*12)]) ) {
#if 0
      if (header_buf[1+(j*12)] <= ' ') { /* sample filename[0] */
#endif
        discard = 1;
        break;
      }
      j++;
    }
    if (!discard) {
      /* not very sure, but let's assume GTS */
      assumption[i] = MUS_GTS; i++;
    }
  }
  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* text: NO endian, each byte with 7 (!) bits */

  if (   (i == 0)                       /* nothing found yet or */
      || ( (i == 1) && (assumption[0] == MUS_GTS) ) /* only gts */
     ) {
    j = discard = 0;
    while (   (j < read_cnt)
           && (!discard)
          ) {
      if (   (header_buf[j] > 127)
          || (   (header_buf[j] <  ' ')
              && (header_buf[j] != 0x08) /* <bs>  */
              && (header_buf[j] != 0x09) /* <tab> */
              && (header_buf[j] != 0x0A) /* <lf>  */
              && (header_buf[j] != 0x0D) /* <cr>  */
              && (header_buf[j] != 0x1A) /* <eof> */
             )
         ) {
        discard = 1;
        break;
      }
      j++;
    }
    if ( (!discard) && (j <= read_cnt) ) {
      /* text superseeds GTS */
      assumption[0] = OTHERS_TEXT; i=1;
    }
  } /* i==0 */

  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

  (void) tmp_free ((char *)header_buf);
  return (&assumption[0]);

} /* detect_music_filetype() */

/* ------------------------------------------------------------ */

#ifdef _TEST_

#ifdef __MSC__
# include  <dos.h>
# include  <fcntl.h>
#endif

int main (int argc, char **argv, char **envp) {

  static char   path[128];
  static char   fname[128];
#ifdef __MSC__
  static struct find_t finfo;
  unsigned      fnd;
#endif
  int           idx;
  MUS_T        *mus;
  FILE         *fp;

  if (argv[1]) {
    strcpy (path, argv[1]);
  }
  else {
    strcpy (path, "*.*");
  }

#ifdef __MSC__
  strcpy (fname, path);

  printf ("Looking for \"%s\".\n", fname);

  fnd = _dos_findfirst (fname, _A_NORMAL, &finfo);

  while (fnd == 0) {

    printf ("%-12s : %6ld , ", finfo.name, finfo.size);
#else
    printf ("%-12s : ", path);
#endif

#ifdef __MSC__
    strcpy (fname, finfo.name);
#endif

    fp = fopen (fname, "rb");
#ifdef __MSC__
    setmode (fileno(fp), O_BINARY);
#endif

    /* ------------------------------- */

    mus = detect_music_filetype (fp);

    /* ------------------------------- */

    if (fp) {
      (void) fclose (fp);
    }

    if (!mus) {
      printf ("NULL");
    }
    else {
      idx = 0;
      if (mus[0] == MUS_UNKNOWN) {
        printf ("unknown");
      }
      else {
        while (   (idx < 16)
               && (mus[idx] != MUS_UNKNOWN)
              ) {
          switch (mus[idx]) {
            case MUS_UNKNOWN:   printf ("unknown ");   break;

            case MUS_MOD_OLD15: printf ("MOD-15 ");    break;
            case MUS_MOD_PT4:   printf ("MOD-PT4 ");   break;
            case MUS_MOD_PT6:   printf ("MOD-6CN ");   break;
            case MUS_MOD_PT8:   printf ("MOD-8CN ");   break;

            case MUS_STS:       printf ("STS ");       break;
            case MUS_STM:       printf ("STM ");       break;
            case MUS_GTS:       printf ("GTS ");       break;

            case MUS_MED:       printf ("MED ");       break;

            case MUS_669:       printf ("669 ");       break;
            case MUS_Oktalyser: printf ("Oktalyzer "); break;
            
            case MUS_FORM_SMUS: printf ("IFF-SMUS ");  break;
            case MUS_FORM_TRKR: printf ("IFF-TRKR ");  break;
            case MUS_FORM_MODL: printf ("IFF-MODL ");  break;
            case MUS_FORM_EMOD: printf ("IFF-EMOD ");  break;
            case MUS_MIFF:      printf ("IFF-MIFF ");  break;

            case MUS_TFMX:      printf ("TFMX ");      break;
            case MUS_S3M:       printf ("S3M ");       break;
            case MUS_ULTRA:     printf ("Ultra ");     break;

            case MUS_ROL:       printf ("ROL ");       break;
            case MUS_CMF:       printf ("CTMF ");      break;
            case MUS_SID:       printf ("Sid ");       break;

            case MUS_MMD0:      printf ("MMD0 ");      break;
            case MUS_MMD1:      printf ("MMD1 ");      break;
            case MUS_MIDI:      printf ("MIDI ");      break;

            case OTHERS_NONE:   printf ("<access err> "); break;
            case OTHERS_EMPTY:  printf ("<empty> ");      break;
            case OTHERS_TEXT:   printf ("text ");         break;
            default:            printf ("unknown ");      break;
          }
          idx++;
        }
      }
    } /* mus != 0 */
    printf ("\n");

#ifdef __MSC__
    fnd = _dos_findnext (&finfo);
  }
#endif
}

#endif /* _TEST_ */

/* <eof> detect.c */
