// gvector2.cpp
// Gouraud Vectors 2, this time in Watcom C/C++ 10.0

// Coded by Terry Sznober (aka Tumblin / Bodies In Motion)
// Please see bim16.nfo for more detailed information about
// myself and the rest of the Bodies In Motion crew.

// last modified January 9, 1996

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <math.h>

#include "fixed32.h"
#include "gpalette.h"

//----- constants
#define MAXDEGREES 1024
#define EYE_DISTANCE Int2Fixed(256)
#define MAXVERTICES 600
#define MAXPOLYGONS 300
#define MAXPOLYVERT 20


//----- these are for the gouraud shading routines ----
int ledge_x[200];  // x coordinate for left edge
int ledge_color[200]; // color for left edge

int redge_x[200]; // destination x coordinate for right edge
int redge_color[200]; // color for right edge


//----- these are for the basic graphics routines
int ytable[200]; // so u can do ytable[y]+x (fast), instead of y*320+x (slow)
char *screen=(char *)0x0a0000; // mode 13h screen
char screen_buff[64000]; // 320x200 system ram screen buffer
unsigned char temp_palette[768]; // temporary palette buffer


//----- 16.16 fixedpoint trig lookup tables
Fixedpoint cosine[MAXDEGREES];  // cosine lookup table
Fixedpoint sine[MAXDEGREES];    // sine lookup table

//----- angle and position of 3d object
int xangle,yangle,zangle; // angles of rotation
int xpos,ypos,zpos; // position of object in 3d

//----- erase box for doing dirty rectangles
int eboxtop; // erase box top coordinate (current frame)
int eboxbottom; // erase box bottom coordinate (current frame)
int eboxleft; // erase box left coordinate (current frame)
int eboxright; // erase box right coordinate (current frame)
int eboxtop1; // erase box top coordinate (1 frame ago)
int eboxbottom1; // erase box bottom coordinate (1 frame ago)
int eboxleft1; // erase box left coordinate (1 frame ago)
int eboxright1; // erase box right coordinate (1 frame ago)

// structure of one vertex
typedef struct
{
	Fixedpoint ox,oy,oz; // vertex original coordinates
	Fixedpoint wx,wy,wz; // vertex working  coordinates
	Fixedpoint Nox,Noy,Noz; // vertex normal original coordinates
	Fixedpoint Nwx,Nwy,Nwz; // vertex normal working coordinates
	int sx,sy,sz; // vertex screen coordinates
	int color; // color of vertex normal
} VertexTYPE;

// structure of one polygon
typedef struct
{
	int NumOfVertices; // number of vertices that make up the polygon
	int Vertex[MAXPOLYVERT]; // vertex indices in counter clockwise order
	int zcenter; // for sorting
	Fixedpoint Nox,Noy,Noz; // polygon normals (only used to get vertex normals)
} PolygonTYPE;

// object structure
typedef struct
{
	Fixedpoint Ox,Oy,Oz; // coordinates of object's origin
	int Ax,Ay,Az; // object's rotation angles
	int NumOfVertices; // number of vertices in object
	VertexTYPE Vertex[MAXVERTICES]; // all vertices in object
	int NumOfPolygons; // number of polygons in object
	PolygonTYPE Polygon[MAXPOLYGONS]; // all polygons in object
} ObjectTYPE;

ObjectTYPE Object; // one object

// structure of light source
typedef struct
{
	Fixedpoint x,y,z; // coodinates of light source
	Fixedpoint wx,wy,wz; // working (intermediate) coordinates
	int ax,ay,az; // rotation angles
} LightSourceTYPE;

LightSourceTYPE LightSource; // one light source

int NumOfSortedPolygons; // number of sorted visible polygons
int PolygonOrderList[MAXPOLYGONS]; // list of polygon indices for qsorting


//---------------------- function prototypes ------------------------
int main(void);
void DrawGouraudPolygon(PolygonTYPE *poly);
void GScanEdge(int x1,int y1,int c1,int x2,int y2,int c2);
void WaitVRT(void);
void SetUpScreen(void);
void InitYTable(void);
void InitTrigTables(void);
void LoadObject(void);
void InitNormals(void);
void RotatePoints(void);
void RotateNormals(void);
void DrawObject(void);
void GetPalette(unsigned char *palettebuffer);
void SetPalette(unsigned char *palettebuffer);
void FadeInPalette(unsigned char *palettebuffer,int speed);
void FadeOutPalette(int speed);
void MakePolygonList(void);
void SortPolygonList(void);
int ComparePolygons(const void *a, const void *b);
void Demo(void);
void InitLightSource(void);
void CalculateColor(void);
//-------------------------------------------------------------------


void SetMode13h(void);
#pragma aux SetMode13h = \
		"mov eax,013h " \
		"int 10h" \
		modify [eax];


void SetMode03h(void);
#pragma aux SetMode03h = \
		"mov eax,003h " \
		"int 10h" \
		modify [eax];



void WaitVRT(void)
{
	while(inp(0x3da) & 0x08);
	while(!(inp(0x3da) & 0x08));
}


void InitYTable(void)
{
	int i;
	for(i=0;i<200;i++)
	{
		ytable[i]=i*320;
	}
}


void InitTrigTables(void)
{
	int i;
	for(i=0; i<MAXDEGREES; i++)
	{
		cosine[i]=Float2Fixed(cos((float)i*360/MAXDEGREES * 3.14159265 / 180.0));
		sine[i]  =Float2Fixed(sin((float)i*360/MAXDEGREES * 3.14159265 / 180.0));
	}
}


void SetUpScreen(void)
{
	// set mode 13h screen
	SetMode13h();

	int i;

	// clear screen memory
	for(i=0;i<64000;i++)
	{
		screen[i]=0;
	}
}

//----------------------------------------------------------------------


// initialize normals 


void InitNormals(void)
{
	double x1,x2,x3,y1,y2,y3,z1,z2,z3,xlen,ylen,zlen,length;
	int i,j,k;
	int count;

	// first calculate the polygon normals
	for(i=0;i < Object.NumOfPolygons; i++)
	{
		x1=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[0]].ox);
		x2=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[1]].ox);
		x3=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[2]].ox);
		y1=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[0]].oy);
		y2=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[1]].oy);
		y3=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[2]].oy);
		z1=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[0]].oz);
		z2=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[1]].oz);
		z3=Fixed2Float(Object.Vertex[Object.Polygon[i].Vertex[2]].oz);

		// calculate perpendicular via the cross product of 1st vertex & normal
		xlen = (y1-y2)*(z1-z3) - (z1-z2)*(y1-y3);
		ylen = (z1-z2)*(x1-x3) - (x1-x2)*(z1-z3);
		zlen = (x1-x2)*(y1-y3) - (y1-y2)*(x1-x3);

		// calculate the length of the normal
		length = sqrt(xlen*xlen + ylen*ylen + zlen*zlen);

		// scale it to a unit normal
		Object.Polygon[i].Nox = Float2Fixed(xlen / length);
		Object.Polygon[i].Noy = Float2Fixed(ylen / length);
		Object.Polygon[i].Noz = Float2Fixed(zlen / length);
	}

	// now figure out the vertex normals

	//This will calculate the normals of the vertices for environment mapping
	for(i=0;i < Object.NumOfVertices; i++)
	{
		count=0;
		xlen=0;
		ylen=0;
		zlen=0;

		// find all the polygons that this vertex belongs to
		for(j=0;j<Object.NumOfPolygons;j++)
		{
			for(k=0;k<Object.Polygon[j].NumOfVertices;k++)
			{
				if(Object.Polygon[j].Vertex[k]==i)
				{
					// vertex belongs to this polygon so add it on
					xlen+=Fixed2Float(Object.Polygon[j].Nox);
					ylen+=Fixed2Float(Object.Polygon[j].Noy);
					zlen+=Fixed2Float(Object.Polygon[j].Noz);
					count++;
				}
			}
		}

		// calculate the average normal
		if(count>0)
		{
			xlen=xlen/count;
			ylen=ylen/count;
			zlen=zlen/count;
		}

		// calculate the length of the normal
		length = sqrt(xlen*xlen + ylen*ylen + zlen*zlen);

		// scale it to a unit normal
		Object.Vertex[i].Nox = Float2Fixed(xlen / length);
		Object.Vertex[i].Noy = Float2Fixed(ylen / length);
		Object.Vertex[i].Noz = Float2Fixed(zlen / length);
	}
}



// rotate normals 


void RotateNormals(void)
{
	int i;
	Fixedpoint nx,ny,nz,cosxangle,sinxangle,cosyangle,sinyangle,coszangle,sinzangle;
	VertexTYPE *vert; // pointer to a vertex structure

	// get sine and cosine angles to save time from table lookup in inner loop
	sinxangle=sine[xangle];
	cosxangle=cosine[xangle];
	sinyangle=sine[yangle];
	cosyangle=cosine[yangle];
	sinzangle=sine[zangle];
	coszangle=cosine[zangle];

	for(i=0;i<Object.NumOfVertices;i++)
	{
		vert=&Object.Vertex[i];

		// rotate around the x-axis
		vert->Nwz=FixedMul(vert->Noy , cosxangle) - FixedMul(vert->Noz , sinxangle);
		vert->Nwy=FixedMul(vert->Noy , sinxangle) + FixedMul(vert->Noz , cosxangle);
		vert->Nwx=vert->Nox;

		// rotate around the y-axis
		nx=FixedMul(vert->Nwx , cosyangle) - FixedMul(vert->Nwz , sinyangle);
		nz=FixedMul(vert->Nwx , sinyangle) + FixedMul(vert->Nwz , cosyangle);
		vert->Nwx=nx;
		vert->Nwz=nz;

		// rotate around the z-axis
		nx=FixedMul(vert->Nwx , coszangle) - FixedMul(vert->Nwy , sinzangle);
		ny=FixedMul(vert->Nwx , sinzangle) + FixedMul(vert->Nwy , coszangle);

		// reverse the direction of the normals for lightsource shading
		vert->Nwx=nx;
		vert->Nwy=ny;
		vert->Nwz=nz;
	}
}

//-------------------------- calculate color of vertex normal ----------

void CalculateColor(void)
{
	int i;
	int color;
	Fixedpoint DotProduct;

	for(i=0; i < Object.NumOfVertices; i++)
	{
		DotProduct=FixedMul(Object.Vertex[i].Nwx , LightSource.x) +
							 FixedMul(Object.Vertex[i].Nwy , LightSource.y) +
							 FixedMul(Object.Vertex[i].Nwz , LightSource.z);

		Object.Vertex[i].color = 80 + Fixed2Int(FixedMul(DotProduct,Int2Fixed(64)));
	}
}



//----------------------------------------------------------------------

void RotatePoints(void)
{
	int i;
	Fixedpoint nx,ny,nz,cosxangle,sinxangle,cosyangle,sinyangle,coszangle,sinzangle;

	// get the sine and cosine angles to save time from table lookup
	sinxangle=sine[xangle];
	cosxangle=cosine[xangle];
	sinyangle=sine[yangle];
	cosyangle=cosine[yangle];
	sinzangle=sine[zangle];
	coszangle=cosine[zangle];


	// initialize the erase-box info to the extreme values
	eboxtop=200;
	eboxleft=320;
	eboxright=0;
	eboxbottom=0;


	// loop through all vertices in object
	for(i=0;i<Object.NumOfVertices;i++)
	{
		// make a pointer to the current vertex
		VertexTYPE *vert=&Object.Vertex[i];

		// rotate around the x-axis
		vert->wz=FixedMul(vert->oy,cosxangle) - FixedMul(vert->oz,sinxangle);
		vert->wy=FixedMul(vert->oy,sinxangle) + FixedMul(vert->oz,cosxangle);
		vert->wx=vert->ox;

		// rotate around the y-axis
		nx=FixedMul(vert->wx,cosyangle) - FixedMul(vert->wz,sinyangle);
		nz=FixedMul(vert->wx,sinyangle) + FixedMul(vert->wz,cosyangle);
		vert->wx=nx;
		vert->wz=nz;

		// rotate around the z-axis
		nx=FixedMul(vert->wx,coszangle) - FixedMul(vert->wy,sinzangle);
		ny=FixedMul(vert->wx,sinzangle) + FixedMul(vert->wy,coszangle);
		vert->wx=nx;
		vert->wy=ny;

		// project the 3-D coordinates to screen coordinates
		vert->sx=Fixed2Int(FixedMul(FixedDiv(vert->wx+Int2Fixed(xpos),vert->wz+Int2Fixed(zpos)),Int2Fixed(256))) + 160;
		vert->sy=Fixed2Int(FixedMul(FixedDiv(vert->wy+Int2Fixed(ypos),vert->wz+Int2Fixed(zpos)),Int2Fixed(256))) + 100;
		vert->sz=Fixed2Int(vert->wz+Int2Fixed(zpos));

		// while we are at it, update the erase-box information
		if(vert->sx < eboxleft) eboxleft = vert->sx;
		if(vert->sx > eboxright) eboxright = vert->sx;
		if(vert->sy < eboxtop) eboxtop = vert->sy;
		if(vert->sy > eboxbottom) eboxbottom = vert->sy;
	}

	// increase the size of the dirty rectangle if necessary
	if(eboxleft1>eboxleft) eboxleft1=eboxleft;
	if(eboxright1<eboxright) eboxright1=eboxright;
	if(eboxtop1>eboxtop) eboxtop1=eboxtop;
	if(eboxbottom1<eboxbottom) eboxbottom1=eboxbottom;

	// clip the erase box
	if(eboxleft<0) eboxleft=0;
	if(eboxtop<0) eboxtop=0;
	if(eboxright>319) eboxright=319;
	if(eboxbottom>199) eboxbottom=199;
}

//--------------------------------------------------------------------------

void DrawObject(void)
{
	int i,j;
	char *temp_ptr;
	char *temp_ptr2;

	// wait for vertical retrace
//  WaitVRT();

	// copy the background into screen buffer
	for(i=eboxtop1;i<=eboxbottom1;i++)
	{
		temp_ptr=&screen_buff[ytable[i]+eboxleft1];
		for(j=eboxleft1;j<=eboxright1;j++)
		{
			*temp_ptr=0;
			temp_ptr++;
		}
	}

	MakePolygonList();
	SortPolygonList();

	CalculateColor();

	VertexTYPE *v0,*v1,*v2,*v3;

	for(i=0;i<NumOfSortedPolygons;i++)
	{
		v0=&Object.Vertex[Object.Polygon[PolygonOrderList[i]].Vertex[0]];
		v1=&Object.Vertex[Object.Polygon[PolygonOrderList[i]].Vertex[1]];
		v2=&Object.Vertex[Object.Polygon[PolygonOrderList[i]].Vertex[2]];

		// if expression results in a negative then polygon is visible
		if( ((v1->sx - v0->sx) * (v2->sy - v0->sy) - (v1->sy - v0->sy) * (v2->sx - v0->sx)) < 0 )
		{
			DrawGouraudPolygon(&Object.Polygon[PolygonOrderList[i]]);
		}
	}

	// fire the dirty rectangle to the screen
	for(i=eboxtop1;i<=eboxbottom1;i++)
	{
		temp_ptr=&screen_buff[ytable[i]+eboxleft1];
		temp_ptr2=&screen[ytable[i]+eboxleft1];
		for(j=eboxleft1;j<=eboxright1;j++)
		{
			*temp_ptr2 = *temp_ptr;
			temp_ptr++;
			temp_ptr2++;
		}
	}

	// pass down the erase box information
	eboxtop1=eboxtop;
	eboxbottom1=eboxbottom;
	eboxleft1=eboxleft;
	eboxright1=eboxright;
}

// load object

void LoadObject(void)
{
	FILE *file;
	int i,j;
	short temp;
	int result;
	VertexTYPE *vert; // pointer to a vertex structure
	char filename[80];

	system("dir *.v10/w");
	printf("\nEnter the filename (filename.v10): ");
	gets(filename);


	// open the file to read from it
	if ((file = fopen(filename,"rb")) == NULL)
	{
		printf("\n\nCannot open .V10 object file.\n");
	}
	else
	{
		// okay file is ready to read data from it

		// read number of dots in file
		fread(&temp,sizeof(short),1,file);
		Object.NumOfVertices=(int)temp;
		// read in all of the object's dots
		for(i=0;i < Object.NumOfVertices; i++)
		{
			fread(&temp,sizeof(short),1,file);
			Object.Vertex[i].ox=Int2Fixed(temp);
			fread(&temp,sizeof(short),1,file);
			Object.Vertex[i].oy=Int2Fixed(temp);
			fread(&temp,sizeof(short),1,file);
			Object.Vertex[i].oz=Int2Fixed(temp);
		}

		// read in the number of polygons
		fread(&temp,sizeof(short),1,file);
		Object.NumOfPolygons=(int)temp;
		// read in the information for each polygon
		for(i=0; i < Object.NumOfPolygons; i++)
		{
			// read in the number of vertices in polygon
			fread(&temp,sizeof(short),1,file);
			Object.Polygon[i].NumOfVertices=(int)temp;
			// read in the polygon's vertices (already in counter clockwise order)
			for(j=0; j< Object.Polygon[i].NumOfVertices; j++)
			{
				fread(&temp,sizeof(short),1,file);
				Object.Polygon[i].Vertex[j]=(int)temp;
			}
			// read the color of the polygon
			fread(&temp,sizeof(short),1,file);
		}

		// we're finished, close the file
		fclose(file);
	}
}

//---------------------- draw gouraud polygon ------------------------------

void DrawGouraudPolygon(PolygonTYPE *poly)
{
	int i; // for looping
	char *temp_ptr;

	int x,y; // screen coordinates
	int c; // color
	int mc; // slope of color

	// initialize the edge buffers to their extremes
	for(i=0;i<200;i++)
	{
		ledge_x[i]=320;
		redge_x[i]=0;
	}

	// scan each edge of polygon

	for(i=0;i<poly->NumOfVertices-1;i++)
	{
		GScanEdge(Object.Vertex[poly->Vertex[i]].sx,Object.Vertex[poly->Vertex[i]].sy,Object.Vertex[poly->Vertex[i]].color,
							Object.Vertex[poly->Vertex[i+1]].sx,Object.Vertex[poly->Vertex[i+1]].sy,Object.Vertex[poly->Vertex[i+1]].color);
	}
	GScanEdge(Object.Vertex[poly->Vertex[i]].sx,Object.Vertex[poly->Vertex[i]].sy,Object.Vertex[poly->Vertex[i]].color,
						Object.Vertex[poly->Vertex[0]].sx,Object.Vertex[poly->Vertex[0]].sy,Object.Vertex[poly->Vertex[0]].color);

	// gouraud fill each horizontal scanline
	for(y=0;y<200;y++)
	{
		// if the scanline belongs to the polygon...
		if(ledge_x[y] <= redge_x[y])
		{
			// find the slope of the color
			if( (redge_x[y] - ledge_x[y]) != 0)
			{
				mc = ((redge_color[y] - ledge_color[y]) << 16) /(redge_x[y] - ledge_x[y]);
			}
			else
			{
				mc = (redge_color[y] - ledge_color[y])<<16;
			}

			// initialize color
			c=ledge_color[y]<<16;

			// draw the horizontal scanline
			temp_ptr=&screen_buff[ytable[y]+ledge_x[y]];
			for(x=ledge_x[y];x<redge_x[y];x++)
			{
				*temp_ptr = (char)(c>>16);
				// interpolate color and screen pointer
				c+=mc;
				temp_ptr++;
			}
		}
	}
}


void GScanEdge(int x1,int y1,int c1,int x2,int y2,int c2)
{

	int mx,mc; // slopes of x and color
	int temp; // for swapping
	int x,y,c; // source x and y screen coordinates, and color (all in 16.16)

	// make sure that edge goes from top to bottom
	if(y1 > y2)
	{
		// we need to swap the coordinates around
		temp=x1;
		x1=x2;
		x2=temp;

		temp=y1;
		y1=y2;
		y2=temp;

		temp=c1;
		c1=c2;
		c2=temp;
	}

	// initialize the slopes for stepping the edges
	if((y2-y1) != 0)
	{
		mx = ((x2-x1) << 16) / (y2-y1); // dx/dy
		mc = ((c2-c1) << 16) / (y2-y1); // dc/dy
	}
	else
	{
		mx = ((x2-x1) << 16); // dx
		mc = ((c2-c1) << 16); // dc
	}

	// initialize first coordinates
	x=x1<<16;
	c=c1<<16;

	// step through edge and record coordinates along the way
	for(y=y1;y<y2;y++)
	{
		// update left edge information

		if( ((y>>16)>=0) && ((y>>16)<=199) )
		{
			if( ((x>>16)<ledge_x[y]) && (ledge_x[y]>=0) )
			{
				ledge_x[y]=(x>>16);
				ledge_color[y]=(c>>16);
			}

			if( ((x>>16)>redge_x[y]) && (redge_x[y]<=319) )
			{
				// update right edge information
				redge_x[y]=(x>>16);
				redge_color[y]=(c>>16);
			}
		}

		// increment the coordinates by their respective slopes
		x+=mx;
		c+=mc;
	}
}


//------------------------------- palette functions ---------------

void GetPalette(unsigned char *palettebuffer)
{
	int i;

	for(i=0;i<256;i++)
	{
		outp(0x3c7,i);  // color number to get data from
		palettebuffer[i*3]   = inp(0x3c9);  // red
		palettebuffer[i*3+1] = inp(0x3c9);  // green
		palettebuffer[i*3+2] = inp(0x3c9);  // blue
	}
}

void SetPalette(unsigned char *palettebuffer)
{
	int i;

	for(i=0;i<256;i++)
	{
		outp(0x3c8,i);  // color number to set
		outp(0x3c9,palettebuffer[i*3]);   // red
		outp(0x3c9,palettebuffer[i*3+1]); // green
		outp(0x3c9,palettebuffer[i*3+2]); // blue
	}
}

void FadeInPalette(unsigned char *palettebuffer,int speed)
{
	// palettebuffer is the destination palette you want to reach
	// uses interpolation

	int i,j,k;
	short temppalette[768]={0};
	unsigned char temppalette2[768]={0};
	short delta[768];

	// find out the delta value of each color component for interpolation
	for(i=0;i<768;i++)
	{
		delta[i] = ( ( (short)(palettebuffer[i]) ) << 8 ) /64;
	}


	for(i=0;i<64;i++)
	{
		for(j=0;j<768;j++)
		{
			temppalette[j] += delta[j];
			temppalette2[j] = (unsigned char)(temppalette[j]>>8);
		}
		for(k=0;k<speed;k++)
		{
			WaitVRT();
		}
		SetPalette(temppalette2);
	}
}

void FadeOutPalette(int speed)
{
	int i,j,k;
	short temppalette[768]={0};
	unsigned char temppalette2[768]={0};
	short delta[768];

	GetPalette(temppalette2);

	for(i=0;i<768;i++)
	{
		temppalette[i] = ((short)(temppalette2[i]))<<8;
		delta[i] = (((short)temppalette2[i]) <<8 ) / (-64);
	}

	for(i=0;i<64;i++)
	{
		for(j=0;j<768;j++)
		{
			temppalette[j] += delta[j];
			temppalette2[j] = (unsigned char)(temppalette[j] >> 8);
		}
		for(k=0;k<speed;k++)
		{
			WaitVRT();
		}
		SetPalette(temppalette2);
	}
}




//---------------------------------------------------------------------

int main(void)
{
	SetMode03h();

	printf("Gouraud Shaded Vectors 2\n");
	printf("coded by Tumblin / Bodies In Motion '96\n");

	Demo();

	SetMode03h();

	return(0);
}



//--------------------- MAIN DEMO LOOP ---------------------------

void Demo(void)
{
	char key;
	int i; // for looping

	LoadObject();
	InitNormals();

	SetUpScreen();

	// set palette to black
	for(i=0;i<768;i++)
	{
		temp_palette[i]=0;
	}
	SetPalette(temp_palette);

	// initialize a few things
	InitYTable();
	InitTrigTables();
	InitLightSource();

	// initialize angle and position of object
	xangle=0;
	yangle=0;
	zangle=0;
	xpos=0;
	ypos=0;
	zpos=-1024;


	// draw first frame so that globe won't "pop out of nowhere" (:
	RotatePoints();
	RotateNormals();
	DrawObject();

	FadeInPalette(palette,1);

	do
	{
		xangle+=4; if(xangle >= MAXDEGREES) xangle=0;
		yangle+=6; if(yangle >= MAXDEGREES) yangle=0;
		zangle+=8; if(zangle >= MAXDEGREES) zangle=0;

		RotatePoints();
		RotateNormals();
		DrawObject();
	} while(!kbhit());

	getch();

	FadeOutPalette(1);
}



// make visible polygon list 

void MakePolygonList(void)
{
	int i,j,k;

	VertexTYPE *v0,*v1,*v2;

	j=0;
	for(i=0;i<Object.NumOfPolygons;i++)
	{
		v0=&Object.Vertex[Object.Polygon[i].Vertex[0]];
		v1=&Object.Vertex[Object.Polygon[i].Vertex[1]];
		v2=&Object.Vertex[Object.Polygon[i].Vertex[2]];

		// if expression results in a negative then polygon is visible
		if( ((v1->sx - v0->sx) * (v2->sy - v0->sy) - (v1->sy - v0->sy) * (v2->sx - v0->sx)) < 0 )
		{
			PolygonOrderList[j++]=i;
		}
	}
	NumOfSortedPolygons=j;
}

// sort polygons 


void SortPolygonList(void)
{
	int i,j;
	int maxz,minz;
	int temp;
	PolygonTYPE *poly;

	// first find the distance of each polygon (from midpoint of max & min z's)
	for(i=0;i<Object.NumOfPolygons;i++)
	{

		poly=&Object.Polygon[i];
		minz=65535;
		maxz=-65536;
		for(j=0;j<poly->NumOfVertices;j++)
		{
			if(Object.Vertex[poly->Vertex[j]].sz < minz)
			{
				minz=Object.Vertex[poly->Vertex[j]].sz;
			}
			if(Object.Vertex[poly->Vertex[j]].sz > maxz)
			{
				maxz=Object.Vertex[poly->Vertex[j]].sz;
			}
		}
		// now calculate the distance
		poly->zcenter=(maxz+minz)<<1;
	}

	// qsort the polygons
	qsort(PolygonOrderList,NumOfSortedPolygons,sizeof(int),ComparePolygons);

	// all done the sorting process
}

// compare balls (for qsort) 

int ComparePolygons(const void *a, const void *b)
{
	if( Object.Polygon[*(int *)a].zcenter < Object.Polygon[*(int *)b].zcenter )
	{
		return -1;
	}
	else if( Object.Polygon[*(int *)a].zcenter > Object.Polygon[*(int *)b].zcenter )
	{
		return +1;
	}
	else
	{
		return 0;
	}
}

//-------------------------- inilialize light source -----------------------

void InitLightSource(void)
{
	double xlen,ylen,zlen,length;

	// assign the delault light source position
	xlen=-10;
	ylen=-10;
	zlen=-10;

	// calculate the length of the vector (light source to 0,0,0)
	length = sqrt(xlen*xlen + ylen*ylen + zlen*zlen);

	// scale it to a unit vector
	LightSource.x = Float2Fixed(xlen/length);
	LightSource.y = Float2Fixed(ylen/length);
	LightSource.z = Float2Fixed(zlen/length);
}
