#include "stdafx.h"
#include "blobs.h"

#define POSBORDERVALUE		0.5f
#define NEGBORDERVALUE		0.5f

// kody krawedzi kolejne bity odpowiadaja kolejnym krawedzia 1 na dabym bicie oznacz ze na
// tej krawedzi jest wierzcholek (np 0x1 oznacza ze wierzcholek jest na krawedzi nr 0 bo
// ustawiony jest tylko zerowy bit!) 
//------------------------------------------------------------------------------------------
// w sumie ta tablice bardzo latwo jest wygenerowac
int edgeTable[256] = 
{
	0x0,   0x109, 0x203, 0x30a, 0x406, 0x50f, 0x605, 0x70c,
	0x80c, 0x905, 0xa0f, 0xb06, 0xc0a, 0xd03, 0xe09, 0xf00,
	0x190, 0x99 , 0x393, 0x29a, 0x596, 0x49f, 0x795, 0x69c,
	0x99c, 0x895, 0xb9f, 0xa96, 0xd9a, 0xc93, 0xf99, 0xe90,
	0x230, 0x339, 0x33 , 0x13a, 0x636, 0x73f, 0x435, 0x53c,
	0xa3c, 0xb35, 0x83f, 0x936, 0xe3a, 0xf33, 0xc39, 0xd30,
	0x3a0, 0x2a9, 0x1a3, 0xaa , 0x7a6, 0x6af, 0x5a5, 0x4ac,
	0xbac, 0xaa5, 0x9af, 0x8a6, 0xfaa, 0xea3, 0xda9, 0xca0,
	0x460, 0x569, 0x663, 0x76a, 0x66 , 0x16f, 0x265, 0x36c,
	0xc6c, 0xd65, 0xe6f, 0xf66, 0x86a, 0x963, 0xa69, 0xb60,
	0x5f0, 0x4f9, 0x7f3, 0x6fa, 0x1f6, 0xff , 0x3f5, 0x2fc,
	0xdfc, 0xcf5, 0xfff, 0xef6, 0x9fa, 0x8f3, 0xbf9, 0xaf0,
	0x650, 0x759, 0x453, 0x55a, 0x256, 0x35f, 0x55 , 0x15c,
	0xe5c, 0xf55, 0xc5f, 0xd56, 0xa5a, 0xb53, 0x859, 0x950,
	0x7c0, 0x6c9, 0x5c3, 0x4ca, 0x3c6, 0x2cf, 0x1c5, 0xcc ,
	0xfcc, 0xec5, 0xdcf, 0xcc6, 0xbca, 0xac3, 0x9c9, 0x8c0,
	0x8c0, 0x9c9, 0xac3, 0xbca, 0xcc6, 0xdcf, 0xec5, 0xfcc,
	0xcc , 0x1c5, 0x2cf, 0x3c6, 0x4ca, 0x5c3, 0x6c9, 0x7c0,
	0x950, 0x859, 0xb53, 0xa5a, 0xd56, 0xc5f, 0xf55, 0xe5c,
	0x15c, 0x55 , 0x35f, 0x256, 0x55a, 0x453, 0x759, 0x650,
	0xaf0, 0xbf9, 0x8f3, 0x9fa, 0xef6, 0xfff, 0xcf5, 0xdfc,
	0x2fc, 0x3f5, 0xff , 0x1f6, 0x6fa, 0x7f3, 0x4f9, 0x5f0,
	0xb60, 0xa69, 0x963, 0x86a, 0xf66, 0xe6f, 0xd65, 0xc6c,
	0x36c, 0x265, 0x16f, 0x66 , 0x76a, 0x663, 0x569, 0x460,
	0xca0, 0xda9, 0xea3, 0xfaa, 0x8a6, 0x9af, 0xaa5, 0xbac,
	0x4ac, 0x5a5, 0x6af, 0x7a6, 0xaa , 0x1a3, 0x2a9, 0x3a0,
	0xd30, 0xc39, 0xf33, 0xe3a, 0x936, 0x83f, 0xb35, 0xa3c,
	0x53c, 0x435, 0x73f, 0x636, 0x13a, 0x33 , 0x339, 0x230,
	0xe90, 0xf99, 0xc93, 0xd9a, 0xa96, 0xb9f, 0x895, 0x99c,
	0x69c, 0x795, 0x49f, 0x596, 0x29a, 0x393, 0x99 , 0x190,
	0xf00, 0xe09, 0xd03, 0xc0a, 0xb06, 0xa0f, 0x905, 0x80c,
	0x70c, 0x605, 0x50f, 0x406, 0x30a, 0x203, 0x109, 0x0
};

// zawiera trojkaty ktore sa tworzone dla konkretnego ukladu wierzcholkow 
// znajdujacych sie wewnatrz pola blobow
// trzy kolejne wartosci w kazdej z tablic (tych 16 elemwntowych)
// zawieraja numery krawedzi z ktorych wierzcholki wchodza w sklad triangla
// (np {1, 7, 11... oznacza ze pierwszy face bedzie rozpiety miedzy tymi wlasnei krawedzaimi)
// -1 to end
//------------------------------------------------------------------------------------------
// zeby wygenerowac ta tablice niezla procke by trzeba napisac bo jest w sumie 16 przypadkow...
int triTable[256][16] =
{
	{ -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  3,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  9,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  3,  8,  9,  1,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1, 10,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  3,  8,  1, 10,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  9, 10,  2,  0,  9,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  2,  3,  8,  2,  8, 10, 10,  8,  9, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  2, 11,  8,  0, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  0,  9,  2, 11,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  2, 11,  1, 11,  9,  9, 11,  8, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  1, 10, 11,  3, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  1, 10,  0, 10,  8,  8, 10, 11, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  0,  9,  3,  9, 11, 11,  9, 10, -1, -1, -1, -1, -1, -1, -1 },
	{  9, 10,  8, 10, 11,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  8,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  0,  3,  7,  4,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  9,  1,  8,  7,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  9,  1,  4,  1,  7,  7,  1,  3, -1, -1, -1, -1, -1, -1, -1 },
	{  1, 10,  2,  8,  7,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  7,  4,  3,  4,  0,  1, 10,  2, -1, -1, -1, -1, -1, -1, -1 },
	{  9, 10,  2,  9,  2,  0,  8,  7,  4, -1, -1, -1, -1, -1, -1, -1 },
	{  2,  9, 10,  2,  7,  9,  2,  3,  7,  7,  4,  9, -1, -1, -1, -1 },
	{  8,  7,  4,  3,  2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{ 11,  7,  4, 11,  4,  2,  2,  4,  0, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  1,  0,  8,  7,  4,  2, 11,  3, -1, -1, -1, -1, -1, -1, -1 },
	{  4, 11,  7,  9, 11,  4,  9,  2, 11,  9,  1,  2, -1, -1, -1, -1 },
	{  3,  1, 10,  3, 10, 11,  7,  4,  8, -1, -1, -1, -1, -1, -1, -1 },
	{  1, 10, 11,  1, 11,  4,  1,  4,  0,  7,  4, 11, -1, -1, -1, -1 },
	{  4,  8,  7,  9, 11,  0,  9, 10, 11, 11,  3,  0, -1, -1, -1, -1 },
	{  4, 11,  7,  4,  9, 11,  9, 10, 11, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  4,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  4,  5,  0,  3,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  4,  5,  1,  0,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  8,  4,  5,  8,  5,  3,  3,  5,  1, -1, -1, -1, -1, -1, -1, -1 },
	{  1, 10,  2,  9,  4,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  8,  0,  1, 10,  2,  4,  5,  9, -1, -1, -1, -1, -1, -1, -1 },
	{  5, 10,  2,  5,  2,  4,  4,  2,  0, -1, -1, -1, -1, -1, -1, -1 },
	{  2,  5, 10,  3,  5,  2,  3,  4,  5,  3,  8,  4, -1, -1, -1, -1 },
	{  9,  4,  5,  2, 11,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  2, 11,  0, 11,  8,  4,  5,  9, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  4,  5,  0,  5,  1,  2, 11,  3, -1, -1, -1, -1, -1, -1, -1 },
	{  2,  5,  1,  2,  8,  5,  2, 11,  8,  4,  5,  8, -1, -1, -1, -1 },
	{ 10, 11,  3, 10,  3,  1,  9,  4,  5, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  5,  9,  0,  1,  8,  8,  1, 10,  8, 10, 11, -1, -1, -1, -1 },
	{  5,  0,  4,  5, 11,  0,  5, 10, 11, 11,  3,  0, -1, -1, -1, -1 },
	{  5,  8,  4,  5, 10,  8, 10, 11,  8, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  8,  7,  5,  9,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  0,  3,  9,  3,  5,  5,  3,  7, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  8,  7,  0,  7,  1,  1,  7,  5, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  3,  5,  3,  7,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  8,  7,  9,  7,  5, 10,  2,  1, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  2,  1,  9,  0,  5,  5,  0,  3,  5,  3,  7, -1, -1, -1, -1 },
	{  8,  2,  0,  8,  5,  2,  8,  7,  5, 10,  2,  5, -1, -1, -1, -1 },
	{  2,  5, 10,  2,  3,  5,  3,  7,  5, -1, -1, -1, -1, -1, -1, -1 },
	{  7,  5,  9,  7,  9,  8,  3,  2, 11, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  7,  5,  9,  2,  7,  9,  0,  2,  2, 11,  7, -1, -1, -1, -1 },
	{  2, 11,  3,  0,  8,  1,  1,  8,  7,  1,  7,  5, -1, -1, -1, -1 },
	{ 11,  1,  2, 11,  7,  1,  7,  5,  1, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  8,  5,  8,  7,  5, 10,  3,  1, 10, 11,  3, -1, -1, -1, -1 },
	{  5,  0,  7,  5,  9,  0,  7,  0, 11,  1, 10,  0, 11,  0, 10, -1 },
	{ 11,  0, 10, 11,  3,  0, 10,  0,  5,  8,  7,  0,  5,  0,  7, -1 },
	{ 11,  5, 10,  7,  5, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  5,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  3,  8,  5,  6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  1,  0,  5,  6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  3,  8,  1,  8,  9,  5,  6, 10, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  5,  6,  2,  1,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  5,  6,  1,  6,  2,  3,  8,  0, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  5,  6,  9,  6,  0,  0,  6,  2, -1, -1, -1, -1, -1, -1, -1 },
	{  5,  8,  9,  5,  2,  8,  5,  6,  2,  3,  8,  2, -1, -1, -1, -1 },
	{  2, 11,  3, 10,  5,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{ 11,  8,  0, 11,  0,  2, 10,  5,  6, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  9,  1,  2, 11,  3,  5,  6, 10, -1, -1, -1, -1, -1, -1, -1 },
	{  5,  6, 10,  1,  2,  9,  9,  2, 11,  9, 11,  8, -1, -1, -1, -1 },
	{  6, 11,  3,  6,  3,  5,  5,  3,  1, -1, -1, -1, -1, -1, -1, -1 },
	{  0, 11,  8,  0,  5, 11,  0,  1,  5,  5,  6, 11, -1, -1, -1, -1 },
	{  3,  6, 11,  0,  6,  3,  0,  5,  6,  0,  9,  5, -1, -1, -1, -1 },
	{  6,  9,  5,  6, 11,  9, 11,  8,  9, -1, -1, -1, -1, -1, -1, -1 },
	{  5,  6, 10,  4,  8,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  0,  3,  4,  3,  7,  6, 10,  5, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  0,  9,  5,  6, 10,  8,  7,  4, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  5,  6,  1,  7,  9,  1,  3,  7,  7,  4,  9, -1, -1, -1, -1 },
	{  6,  2,  1,  6,  1,  5,  4,  8,  7, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  5,  2,  5,  6,  2,  3,  4,  0,  3,  7,  4, -1, -1, -1, -1 },
	{  8,  7,  4,  9,  5,  0,  0,  5,  6,  0,  6,  2, -1, -1, -1, -1 },
	{  7,  9,  3,  7,  4,  9,  3,  9,  2,  5,  6,  9,  2,  9,  6, -1 },
	{  3,  2, 11,  7,  4,  8, 10,  5,  6, -1, -1, -1, -1, -1, -1, -1 },
	{  5,  6, 10,  4,  2,  7,  4,  0,  2,  2, 11,  7, -1, -1, -1, -1 },
	{  0,  9,  1,  4,  8,  7,  2, 11,  3,  5,  6, 10, -1, -1, -1, -1 },
	{  9,  1,  2,  9,  2, 11,  9, 11,  4,  7,  4, 11,  5,  6, 10, -1 },
	{  8,  7,  4,  3,  5, 11,  3,  1,  5,  5,  6, 11, -1, -1, -1, -1 },
	{  5, 11,  1,  5,  6, 11,  1, 11,  0,  7,  4, 11,  0, 11,  4, -1 },
	{  0,  9,  5,  0,  5,  6,  0,  6,  3, 11,  3,  6,  8,  7,  4, -1 },
	{  6,  9,  5,  6, 11,  9,  4,  9,  7,  7,  9, 11, -1, -1, -1, -1 },
	{ 10,  9,  4,  6, 10,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  6, 10,  4, 10,  9,  0,  3,  8, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  1,  0, 10,  0,  6,  6,  0,  4, -1, -1, -1, -1, -1, -1, -1 },
	{  8,  1,  3,  8,  6,  1,  8,  4,  6,  6, 10,  1, -1, -1, -1, -1 },
	{  1,  9,  4,  1,  4,  2,  2,  4,  6, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  8,  0,  1,  9,  2,  2,  9,  4,  2,  4,  6, -1, -1, -1, -1 },
	{  0,  4,  2,  4,  6,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  8,  2,  3,  8,  4,  2,  4,  6,  2, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  9,  4, 10,  4,  6, 11,  3,  2, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  2,  8,  2, 11,  8,  4, 10,  9,  4,  6, 10, -1, -1, -1, -1 },
	{  3,  2, 11,  0,  6,  1,  0,  4,  6,  6, 10,  1, -1, -1, -1, -1 },
	{  6,  1,  4,  6, 10,  1,  4,  1,  8,  2, 11,  1,  8,  1, 11, -1 },
	{  9,  4,  6,  9,  6,  3,  9,  3,  1, 11,  3,  6, -1, -1, -1, -1 },
	{  8,  1, 11,  8,  0,  1, 11,  1,  6,  9,  4,  1,  6,  1,  4, -1 },
	{  3,  6, 11,  3,  0,  6,  0,  4,  6, -1, -1, -1, -1, -1, -1, -1 },
	{  6,  8,  4, 11,  8,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  7,  6, 10,  7, 10,  8,  8, 10,  9, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  3,  7,  0,  7, 10,  0, 10,  9,  6, 10,  7, -1, -1, -1, -1 },
	{ 10,  7,  6,  1,  7, 10,  1,  8,  7,  1,  0,  8, -1, -1, -1, -1 },
	{ 10,  7,  6, 10,  1,  7,  1,  3,  7, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  6,  2,  1,  8,  6,  1,  9,  8,  8,  7,  6, -1, -1, -1, -1 },
	{  2,  9,  6,  2,  1,  9,  6,  9,  7,  0,  3,  9,  7,  9,  3, -1 },
	{  7,  0,  8,  7,  6,  0,  6,  2,  0, -1, -1, -1, -1, -1, -1, -1 },
	{  7,  2,  3,  6,  2,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  2, 11,  3, 10,  8,  6, 10,  9,  8,  8,  7,  6, -1, -1, -1, -1 },
	{  2,  7,  0,  2, 11,  7,  0,  7,  9,  6, 10,  7,  9,  7, 10, -1 },
	{  1,  0,  8,  1,  8,  7,  1,  7, 10,  6, 10,  7,  2, 11,  3, -1 },
	{ 11,  1,  2, 11,  7,  1, 10,  1,  6,  6,  1,  7, -1, -1, -1, -1 },
	{  8,  6,  9,  8,  7,  6,  9,  6,  1, 11,  3,  6,  1,  6,  3, -1 },
	{  0,  1,  9, 11,  7,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  7,  0,  8,  7,  6,  0,  3,  0, 11, 11,  0,  6, -1, -1, -1, -1 },
	{  7,  6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  7, 11,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  8,  0, 11,  6,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  9,  1, 11,  6,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  8,  9,  1,  8,  1,  3, 11,  6,  7, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  2,  1,  6,  7, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1, 10,  2,  3,  8,  0,  6,  7, 11, -1, -1, -1, -1, -1, -1, -1 },
	{  2,  0,  9,  2,  9, 10,  6,  7, 11, -1, -1, -1, -1, -1, -1, -1 },
	{  6,  7, 11,  2,  3, 10, 10,  3,  8, 10,  8,  9, -1, -1, -1, -1 },
	{  7,  3,  2,  6,  7,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  7,  8,  0,  7,  0,  6,  6,  0,  2, -1, -1, -1, -1, -1, -1, -1 },
	{  2,  6,  7,  2,  7,  3,  0,  9,  1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  2,  6,  1,  6,  8,  1,  8,  9,  8,  6,  7, -1, -1, -1, -1 },
	{ 10,  6,  7, 10,  7,  1,  1,  7,  3, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  6,  7,  1, 10,  7,  1,  7,  8,  1,  8,  0, -1, -1, -1, -1 },
	{  0,  7,  3,  0, 10,  7,  0,  9, 10,  6,  7, 10, -1, -1, -1, -1 },
	{  7, 10,  6,  7,  8, 10,  8,  9, 10, -1, -1, -1, -1, -1, -1, -1 },
	{  6,  4,  8, 11,  6,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  3, 11,  6,  3,  6,  0,  0,  6,  4, -1, -1, -1, -1, -1, -1, -1 },
	{  8, 11,  6,  8,  6,  4,  9,  1,  0, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  6,  4,  9,  3,  6,  9,  1,  3, 11,  6,  3, -1, -1, -1, -1 },
	{  6,  4,  8,  6,  8, 11,  2,  1, 10, -1, -1, -1, -1, -1, -1, -1 },
	{  1, 10,  2,  3, 11,  0,  0, 11,  6,  0,  6,  4, -1, -1, -1, -1 },
	{  4,  8, 11,  4, 11,  6,  0,  9,  2,  2,  9, 10, -1, -1, -1, -1 },
	{ 10,  3,  9, 10,  2,  3,  9,  3,  4, 11,  6,  3,  4,  3,  6, -1 },
	{  8,  3,  2,  8,  2,  4,  4,  2,  6, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  2,  4,  4,  2,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  0,  9,  2,  4,  3,  2,  6,  4,  4,  8,  3, -1, -1, -1, -1 },
	{  1,  4,  9,  1,  2,  4,  2,  6,  4, -1, -1, -1, -1, -1, -1, -1 },
	{  8,  3,  1,  8,  1,  6,  8,  6,  4,  6,  1, 10, -1, -1, -1, -1 },
	{ 10,  0,  1, 10,  6,  0,  6,  4,  0, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  3,  6,  4,  8,  3,  6,  3, 10,  0,  9,  3, 10,  3,  9, -1 },
	{ 10,  4,  9,  6,  4, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  5,  9,  7, 11,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  3,  8,  4,  5,  9, 11,  6,  7, -1, -1, -1, -1, -1, -1, -1 },
	{  5,  1,  0,  5,  0,  4,  7, 11,  6, -1, -1, -1, -1, -1, -1, -1 },
	{ 11,  6,  7,  8,  4,  3,  3,  4,  5,  3,  5,  1, -1, -1, -1, -1 },
	{  9,  4,  5, 10,  2,  1,  7, 11,  6, -1, -1, -1, -1, -1, -1, -1 },
	{  6,  7, 11,  1, 10,  2,  0,  3,  8,  4,  5,  9, -1, -1, -1, -1 },
	{  7, 11,  6,  5, 10,  4,  4, 10,  2,  4,  2,  0, -1, -1, -1, -1 },
	{  3,  8,  4,  3,  4,  5,  3,  5,  2, 10,  2,  5, 11,  6,  7, -1 },
	{  7,  3,  2,  7,  2,  6,  5,  9,  4, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  4,  5,  0,  6,  8,  0,  2,  6,  6,  7,  8, -1, -1, -1, -1 },
	{  3,  2,  6,  3,  6,  7,  1,  0,  5,  5,  0,  4, -1, -1, -1, -1 },
	{  6,  8,  2,  6,  7,  8,  2,  8,  1,  4,  5,  8,  1,  8,  5, -1 },
	{  9,  4,  5, 10,  6,  1,  1,  6,  7,  1,  7,  3, -1, -1, -1, -1 },
	{  1, 10,  6,  1,  6,  7,  1,  7,  0,  8,  0,  7,  9,  4,  5, -1 },
	{  4, 10,  0,  4,  5, 10,  0, 10,  3,  6,  7, 10,  3, 10,  7, -1 },
	{  7, 10,  6,  7,  8, 10,  5, 10,  4,  4, 10,  8, -1, -1, -1, -1 },
	{  6,  5,  9,  6,  9, 11, 11,  9,  8, -1, -1, -1, -1, -1, -1, -1 },
	{  3, 11,  6,  0,  3,  6,  0,  6,  5,  0,  5,  9, -1, -1, -1, -1 },
	{  0,  8, 11,  0, 11,  5,  0,  5,  1,  5, 11,  6, -1, -1, -1, -1 },
	{  6,  3, 11,  6,  5,  3,  5,  1,  3, -1, -1, -1, -1, -1, -1, -1 },
	{  1, 10,  2,  9, 11,  5,  9,  8, 11, 11,  6,  5, -1, -1, -1, -1 },
	{  0,  3, 11,  0, 11,  6,  0,  6,  9,  5,  9,  6,  1, 10,  2, -1 },
	{ 11,  5,  8, 11,  6,  5,  8,  5,  0, 10,  2,  5,  0,  5,  2, -1 },
	{  6,  3, 11,  6,  5,  3,  2,  3, 10, 10,  3,  5, -1, -1, -1, -1 },
	{  5,  9,  8,  5,  8,  2,  5,  2,  6,  3,  2,  8, -1, -1, -1, -1 },
	{  9,  6,  5,  9,  0,  6,  0,  2,  6, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  8,  5,  1,  0,  8,  5,  8,  6,  3,  2,  8,  6,  8,  2, -1 },
	{  1,  6,  5,  2,  6,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  6,  3,  1, 10,  6,  3,  6,  8,  5,  9,  6,  8,  6,  9, -1 },
	{ 10,  0,  1, 10,  6,  0,  9,  0,  5,  5,  0,  6, -1, -1, -1, -1 },
	{  0,  8,  3,  5, 10,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  6,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{ 11, 10,  5,  7, 11,  5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{ 11, 10,  5, 11,  5,  7,  8,  0,  3, -1, -1, -1, -1, -1, -1, -1 },
	{  5,  7, 11,  5, 11, 10,  1,  0,  9, -1, -1, -1, -1, -1, -1, -1 },
	{ 10,  5,  7, 10,  7, 11,  9,  1,  8,  8,  1,  3, -1, -1, -1, -1 },
	{ 11,  2,  1, 11,  1,  7,  7,  1,  5, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  3,  8,  1,  7,  2,  1,  5,  7,  7, 11,  2, -1, -1, -1, -1 },
	{  9,  5,  7,  9,  7,  2,  9,  2,  0,  2,  7, 11, -1, -1, -1, -1 },
	{  7,  2,  5,  7, 11,  2,  5,  2,  9,  3,  8,  2,  9,  2,  8, -1 },
	{  2, 10,  5,  2,  5,  3,  3,  5,  7, -1, -1, -1, -1, -1, -1, -1 },
	{  8,  0,  2,  8,  2,  5,  8,  5,  7, 10,  5,  2, -1, -1, -1, -1 },
	{  9,  1,  0,  5,  3, 10,  5,  7,  3,  3,  2, 10, -1, -1, -1, -1 },
	{  9,  2,  8,  9,  1,  2,  8,  2,  7, 10,  5,  2,  7,  2,  5, -1 },
	{  1,  5,  3,  3,  5,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  7,  8,  0,  1,  7,  1,  5,  7, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  3,  0,  9,  5,  3,  5,  7,  3, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  7,  8,  5,  7,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  5,  4,  8,  5,  8, 10, 10,  8, 11, -1, -1, -1, -1, -1, -1, -1 },
	{  5,  4,  0,  5,  0, 11,  5, 11, 10, 11,  0,  3, -1, -1, -1, -1 },
	{  0,  9,  1,  8, 10,  4,  8, 11, 10, 10,  5,  4, -1, -1, -1, -1 },
	{ 10,  4, 11, 10,  5,  4, 11,  4,  3,  9,  1,  4,  3,  4,  1, -1 },
	{  2,  1,  5,  2,  5,  8,  2,  8, 11,  4,  8,  5, -1, -1, -1, -1 },
	{  0, 11,  4,  0,  3, 11,  4, 11,  5,  2,  1, 11,  5, 11,  1, -1 },
	{  0,  5,  2,  0,  9,  5,  2,  5, 11,  4,  8,  5, 11,  5,  8, -1 },
	{  9,  5,  4,  2,  3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  2, 10,  5,  3,  2,  5,  3,  5,  4,  3,  4,  8, -1, -1, -1, -1 },
	{  5,  2, 10,  5,  4,  2,  4,  0,  2, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  2, 10,  3, 10,  5,  3,  5,  8,  4,  8,  5,  0,  9,  1, -1 },
	{  5,  2, 10,  5,  4,  2,  1,  2,  9,  9,  2,  4, -1, -1, -1, -1 },
	{  8,  5,  4,  8,  3,  5,  3,  1,  5, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  5,  4,  1,  5,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  8,  5,  4,  8,  3,  5,  9,  5,  0,  0,  5,  3, -1, -1, -1, -1 },
	{  9,  5,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  7, 11,  4, 11,  9,  9, 11, 10, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  3,  8,  4,  7,  9,  9,  7, 11,  9, 11, 10, -1, -1, -1, -1 },
	{  1, 11, 10,  1,  4, 11,  1,  0,  4,  7, 11,  4, -1, -1, -1, -1 },
	{  3,  4,  1,  3,  8,  4,  1,  4, 10,  7, 11,  4, 10,  4, 11, -1 },
	{  4,  7, 11,  9,  4, 11,  9, 11,  2,  9,  2,  1, -1, -1, -1, -1 },
	{  9,  4,  7,  9,  7, 11,  9, 11,  1,  2,  1, 11,  0,  3,  8, -1 },
	{ 11,  4,  7, 11,  2,  4,  2,  0,  4, -1, -1, -1, -1, -1, -1, -1 },
	{ 11,  4,  7, 11,  2,  4,  8,  4,  3,  3,  4,  2, -1, -1, -1, -1 },
	{  2, 10,  9,  2,  9,  7,  2,  7,  3,  7,  9,  4, -1, -1, -1, -1 },
	{  9,  7, 10,  9,  4,  7, 10,  7,  2,  8,  0,  7,  2,  7,  0, -1 },
	{  3, 10,  7,  3,  2, 10,  7, 10,  4,  1,  0, 10,  4, 10,  0, -1 },
	{  1,  2, 10,  8,  4,  7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  1,  9,  4,  7,  1,  7,  3,  1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  1,  9,  4,  7,  1,  0,  1,  8,  8,  1,  7, -1, -1, -1, -1 },
	{  4,  3,  0,  7,  3,  4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  4,  7,  8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  8, 10, 10,  8, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  9,  0,  3, 11,  9, 11, 10,  9, -1, -1, -1, -1, -1, -1, -1 },
	{  0, 10,  1,  0,  8, 10,  8, 11, 10, -1, -1, -1, -1, -1, -1, -1 },
	{  3, 10,  1, 11, 10,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1, 11,  2,  1,  9, 11,  9,  8, 11, -1, -1, -1, -1, -1, -1, -1 },
	{  3,  9,  0,  3, 11,  9,  1,  9,  2,  2,  9, 11, -1, -1, -1, -1 },
	{  0, 11,  2,  8, 11,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  3, 11,  2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  2,  8,  3,  2, 10,  8, 10,  9,  8, -1, -1, -1, -1, -1, -1, -1 },
	{  9,  2, 10,  0,  2,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  2,  8,  3,  2, 10,  8,  0,  8,  1,  1,  8, 10, -1, -1, -1, -1 },
	{  1,  2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  1,  8,  3,  9,  8,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  1,  9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{  0,  8,  3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 },
	{ -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 }
};

inline 
D3DVECTOR Interpolate( FLOAT hull, he3d_CVector& p1, he3d_CVector& p2, FLOAT valp1, FLOAT valp2 )
{   
	FLOAT			mu;   
	D3DVECTOR		p;   
	
	mu = (hull - valp1) / (valp2 - valp1);
	
	p.x = p1.x + mu*( p2.x - p1.x );
	p.y = p1.y + mu*( p2.y - p1.y );   
	p.z = p1.z + mu*( p2.z - p1.z );
	
	return p;
}

inline
FLOAT CBlob3D::BlobFunction( he3d_CVector& pos )
{
	he3d_CVector	v = pos - m_vPos;
	
	FLOAT			d = Length( v );
	FLOAT			coeff = d/m_fRadius;

	FLOAT			density = 0;
	
	if( coeff <= 0.33f )
		density = m_fDensity*( 1.0f - 3.0f*coeff*coeff );
	else if( coeff <= 1.0f )
		density = 1.5f*m_fDensity*(1.0f - coeff)*(1.0f - coeff);	

	if( m_dwSign ) 
		density = -density;

	return density;	
}

#define	INDEX( a, b, c ) ( (a)*m_dwGridCount*m_dwGridCount + (b)*m_dwGridCount + (c) )

CBlobSystem::CBlobSystem( DWORD dwDensity )
{
	if( dwDensity > 100 )
		dwDensity = 100;

	root			= NULL;
	m_dwGridDensity = dwDensity;
	m_pfGrid		= new FLOAT[dwDensity*dwDensity*dwDensity];	
	m_pRow			= new WORD[3*dwDensity];
	m_pPlane		= new WORD[2*dwDensity*dwDensity];	
		
	m_pvbVertices	= NULL;
	m_pibIndices	= NULL;	
}

BOOL CBlobSystem::Initialize( PDIRECT3DDEVICE8 pDevice )
{			
	if( FAILED( pDevice->CreateVertexBuffer( 10000*sizeof(BLOBVERTEX), D3DUSAGE_DYNAMIC, BLOBFVF, D3DPOOL_DEFAULT, &m_pvbVertices ) ) )
		return FALSE;
		
	if( FAILED( pDevice->CreateIndexBuffer( 30000*sizeof(WORD), D3DUSAGE_DYNAMIC, D3DFMT_INDEX16, D3DPOOL_DEFAULT, &m_pibIndices ) ) )
		return FALSE;

	return TRUE;
}

CBlobSystem::~CBlobSystem()
{
	if( m_pfGrid )
		delete m_pfGrid;
	
	if( m_pRow )
		delete m_pRow;
	
	if( m_pPlane )
		delete m_pPlane;	

	for( CNode* tmp = root ; tmp ; )
	{
		root = tmp->next;
		delete tmp;
		tmp = root;
	}

	if( m_pvbVertices )
		m_pvbVertices->Release();

	if( m_pibIndices )
		m_pibIndices->Release();
}

VOID CBlobSystem::CalculateDensity()
{
	if( !m_dwBlobCount )
		return;

	ResizeToFit();

	FLOAT			startx = m_vCenter.x - 0.5f*m_vSize.x;
	FLOAT			starty = m_vCenter.y + 0.5f*m_vSize.y;
	FLOAT			startz = m_vCenter.z - 0.5f*m_vSize.z;
	FLOAT			deltax = m_vSize.x/(FLOAT)(m_dwGridDensity-1);	
	FLOAT			deltay = m_vSize.y/(FLOAT)(m_dwGridDensity-1);	
	FLOAT			deltaz = m_vSize.z/(FLOAT)(m_dwGridDensity-1);		

	he3d_CVector	vec;
	
	FLOAT			density;

	CNode*			tmp;

	FLOAT*			pGrid = m_pfGrid;
	DWORD			i, j, k;

	for( i = 0, vec.x = startx ; i < m_dwGridDensity ; i++, vec.x+=deltax )
		for( j = 0, vec.y = starty ; j < m_dwGridDensity ; j++, vec.y-= deltay )
			for( k = 0, vec.z = startz ; k < m_dwGridDensity ; k++, vec.z+= deltaz, pGrid++ )
			{				
				tmp		= root;	
				density = 0;
	
				for( ; tmp ; tmp = tmp->next )
				{				
					density += tmp->blob->BlobFunction( vec );
				}

				*pGrid = density;
			}			
}

VOID CBlobSystem::SetGridDensity( DWORD dwDensity )
{
	if( dwDensity > 100 )
		dwDensity = 100;

	if( dwDensity == m_dwGridDensity )
		return;	

	if( m_pfGrid )
		delete m_pfGrid;
	
	if( m_pRow )
		delete m_pRow;
	
	if( m_pPlane )
		delete m_pPlane;		

	m_pfGrid	= new FLOAT[dwDensity*dwDensity*dwDensity];	
	m_pRow		= new WORD[3*dwDensity];
	m_pPlane	= new WORD[2*dwDensity*dwDensity];	
}

DWORD CBlobSystem::GetGridDensity()
{
	return m_dwGridDensity;
}

VOID CBlobSystem::AddBlob( CBlob3D* blob )
{
	CNode*			newblob = new CNode;
	CNode*			tmp;

	newblob->blob = blob;

	if( !root )
	{
		root = newblob;
		root->prev = NULL;
		root->next = NULL;

		m_dwBlobCount = 1;
		
		return;
	}

	for( tmp = root ; tmp->next ; tmp = tmp->next );

	tmp->next = newblob;
	newblob->prev = tmp;
	newblob->next = NULL;

	m_dwBlobCount++;
}

VOID CBlobSystem::RemoveBlob( CBlob3D* blob )
{
	if( !blob )
		return;

	CNode*			tmp = root;

	if( tmp->blob == blob )
	{
		root = tmp->next;
		root->prev = NULL;

		m_dwBlobCount--;

		delete tmp;
	}
	else
	{
		for( ; tmp && tmp->blob != blob ; tmp = tmp->next );

		if( !tmp )
			return;
		
		tmp->prev->next = tmp->next;
		tmp->next->prev = tmp->prev;

		m_dwBlobCount--;

		delete tmp;
	}
}

CBlob3D* CBlobSystem::GetBlob( DWORD index )
{
	CNode*			tmp = root;

	for( DWORD i = 0 ; tmp && i < index ; i++, tmp = tmp->next );

	if( !tmp )
		return NULL;

	return tmp->blob;
}

VOID CBlobSystem::RemoveBlob( DWORD index )
{
	RemoveBlob( GetBlob( index ) );
}

DWORD CBlobSystem::GetBlobCount()
{
	return m_dwBlobCount;
}

#define CLAMP( x ) ( ( (x)>255 ) ? 255 : ( (x)< 0 ) ? 0 : (x) )

#pragma warning( disable : 4244 )

////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	 				   INDEXACJA WIERZCHOLKOW I KRAWEDZI W MARCHING CUBE
//
//
//					 *-----------*						*----4------*
//			        /|4         /|5					   /|          /|
//				   / |         / |					  7 |         5 | 
//				  /	 |        /  |					 /	8        /  9
//				 *-----------*   |					*-----6-----*   |
//				 |7	 *-------|6--*				    |	*----0--|---*
//				 |  / 0      |  / 1				   11  /       10  /
//				 | /         | /					| 3         | 1
//				 |/          |/						|/          |/
//				 *-----------*						*-----2-----* 
//			      3           2
//
VOID CBlobSystem::Triangulate( PDIRECT3DVERTEXBUFFER8 vb, PDIRECT3DINDEXBUFFER8 ib, DWORD& dwVertsCount, DWORD& dwFacesCount, DWORD dwSign )
{							
	BYTE			index;
	WORD			edge;	

	DWORD			dwIndsCount = 0;

	BLOBVERTEX*		pVertTable;
	WORD*			pInd;

	// to jest indexowane numerem wierzcholka
	FLOAT			fGridVal[8];
	he3d_CVector	dvGridVert[4];

	// to jest indexowane numerem krawedzi i zawiera numer vertexa (indices)
	// znajdujacy sie na tej krawedzi
	WORD			wGridIndices[12];	

	ZeroMemory( wGridIndices, sizeof(WORD)*12 );

	DWORD			dwPlaneCount = m_dwGridDensity*m_dwGridDensity;	
	DWORD			planeIndex;

	FLOAT			x, startx = m_vCenter.x - 0.5f*m_vSize.x;
	FLOAT			y, starty = m_vCenter.y + 0.5f*m_vSize.y;
	FLOAT			z, startz = m_vCenter.z - 0.5f*m_vSize.z;
	FLOAT			deltax = m_vSize.x/(FLOAT)(m_dwGridDensity-1);	
	FLOAT			deltay = m_vSize.y/(FLOAT)(m_dwGridDensity-1);	
	FLOAT			deltaz = m_vSize.z/(FLOAT)(m_dwGridDensity-1);	

	FLOAT*			pGrid = m_pfGrid;	
	FLOAT*			pTmpGrid = m_pfGrid;

	DWORD			i, j, k, l, mask;	

	DWORD			offset[] =	{ m_dwGridDensity + 1, 
								  dwPlaneCount + m_dwGridDensity + 1, 
								  dwPlaneCount + m_dwGridDensity,
								  m_dwGridDensity, 
								  1, 
								  dwPlaneCount + 1, 
								  dwPlaneCount, 
								  0 };
	FLOAT			hullValue;		

	dwVertsCount = 0;
	dwFacesCount = 0;

	if( dwSign )
		hullValue = -NEGBORDERVALUE;
	else
		hullValue = POSBORDERVALUE;

	vb->Lock( 0, NULL, (LPBYTE*)&pVertTable, 0 );
	ib->Lock( 0, NULL, (LPBYTE*)&pInd, 0 );

	// pierwsza "plaszczyzna" siatki!!!
	for( x = startx, y = starty, j = 0 ; j < m_dwGridDensity-1 ; j++, y-=deltay, pTmpGrid++ )	
		for( z = startz, k = 0 ; k < m_dwGridDensity-1 ; k++, z+=deltaz, pTmpGrid++ )
		{					
			if( !dwSign )
			{
				for( l = 0, mask = 1, index = 0 ; l < 8 ; l++, mask <<=1 )
					if( ( fGridVal[l] = *( pTmpGrid + offset[l] ) ) >= POSBORDERVALUE )
						index |= mask;
			}
			else
			{
				for( l = 0, mask = 1, index = 0 ; l < 8 ; l++, mask <<=1 )
					if( ( fGridVal[l] = - *( pTmpGrid + offset[l] ) ) >= NEGBORDERVALUE )
						index |= mask;
			}				
		
			if( ( edge = edgeTable[index] ) != 0 )
			{	
				planeIndex = j*m_dwGridDensity + k;

				dvGridVert[0] = he3d_CVector( x, y - deltay, z + deltaz );
				dvGridVert[1] = he3d_CVector( x, y - deltay, z );
				dvGridVert[2] = he3d_CVector( x, y, z + deltaz );

				if( edge&8 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[0], dvGridVert[1], fGridVal[0], fGridVal[3] );
					m_pPlane[planeIndex] = dwVertsCount++;					
				}

				if( edge&256 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[0], dvGridVert[2], fGridVal[0], fGridVal[4] );
					m_pPlane[planeIndex + dwPlaneCount] = dwVertsCount++;
				}				
			}
		}	

	// "plaszczyzny"
	for( x = startx, i = 0 ; i < m_dwGridDensity-1 ; i++, x+=deltax, pGrid += m_dwGridDensity )
	{			
		// pierwszy "rzad" kazdej "plaszczyzny"
		for( y = starty, z = startz, k = 0, pTmpGrid = pGrid ; k < m_dwGridDensity-1 ; k++, z+=deltaz, pTmpGrid++ )
		{					
			if( !dwSign )
			{
				for( l = 0, mask = 1, index = 0 ; l < 8 ; l++, mask <<=1 )
					if( ( fGridVal[l] = *( pTmpGrid + offset[l] ) ) >= POSBORDERVALUE )
						index |= mask;
			}
			else
			{
				for( l = 0, mask = 1, index = 0 ; l < 8 ; l++, mask <<=1 )
					if( ( fGridVal[l] = - *( pTmpGrid + offset[l] ) ) >= NEGBORDERVALUE )
						index |= mask;
			}				
		
			if( ( edge = edgeTable[index] ) != 0 )
			{					
				dvGridVert[0] = he3d_CVector( x, y, z );
				dvGridVert[1] = he3d_CVector( x, y, z + deltaz );
				dvGridVert[2] = he3d_CVector( x + deltax, y, z + deltaz );
				dvGridVert[3] = he3d_CVector( x + deltax, y, z );

				if( edge&16 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[1], dvGridVert[2], fGridVal[4], fGridVal[5] );
					m_pRow[k] = dwVertsCount++;
				}

				if( edge&32 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[2], dvGridVert[3], fGridVal[5], fGridVal[6] );
					m_pRow[k + m_dwGridDensity] = dwVertsCount++;
				}		

				if( edge&128 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[0], dvGridVert[1], fGridVal[7], fGridVal[4] );
					m_pRow[k + (m_dwGridDensity<<1)] = dwVertsCount++;
				}		
			}
		}

		// kolejne "rzedy"
		for( y = starty, j = 0 ; j < m_dwGridDensity-1 ; j++, y-=deltay, pGrid++ )
		{
			// pierwszy klocek!!!
			pTmpGrid = pGrid;
			z = startz;
			
			if( !dwSign )
			{
				for( l = 0, mask = 1, index = 0 ; l < 8 ; l++, mask <<=1 )
					if( ( fGridVal[l] = *( pTmpGrid + offset[l] ) ) >= POSBORDERVALUE )
						index |= mask;
			}
			else
			{
				for( l = 0, mask = 1, index = 0 ; l < 8 ; l++, mask <<=1 )
					if( ( fGridVal[l] = - *( pTmpGrid + offset[l] ) ) >= NEGBORDERVALUE )
						index |= mask;
			}				
		
			if( ( edge = edgeTable[index] ) != 0 )
			{					
				dvGridVert[0] = he3d_CVector( x, y, z );
				dvGridVert[1] = he3d_CVector( x + deltax, y, z );
				dvGridVert[2] = he3d_CVector( x + deltax, y - deltay, z );
				dvGridVert[3] = he3d_CVector( x, y - deltay, z );

				if( edge&4 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[2], dvGridVert[3], fGridVal[2], fGridVal[3] );
					wGridIndices[0] = dwVertsCount++;
				}

				if( edge&64 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[0], dvGridVert[1], fGridVal[7], fGridVal[6] );
					wGridIndices[4] = dwVertsCount++;
				}		

				if( edge&1024 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[1], dvGridVert[2], fGridVal[6], fGridVal[2] );
					wGridIndices[9] = dwVertsCount++;
				}		

				if( edge&2048 )
				{
					pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[0], dvGridVert[3], fGridVal[7], fGridVal[3] );
					wGridIndices[8] = dwVertsCount++;
				}		
			}

			// kolejne "oczka" siatki w "rzedach"
			for( z = startz, k = 0 ; k < m_dwGridDensity-1 ; k++, z+=deltaz, pGrid++ )
			{					
				if( !dwSign )
				{
					for( l = 0, mask = 1, index = 0 ; l < 8 ; l++, mask <<=1 )
						if( ( fGridVal[l] = *( pGrid + offset[l] ) ) >= POSBORDERVALUE )
							index |= mask;
				}
				else
				{
					for( l = 0, mask = 1, index = 0 ; l < 8 ; l++, mask <<=1 )
						if( ( fGridVal[l] = - *( pGrid + offset[l] ) ) >= NEGBORDERVALUE )
							index |= mask;
				}				
			
				if( ( edge = edgeTable[index] ) != 0 )
				{					
					dvGridVert[0] = he3d_CVector( x, y - deltay, z + deltaz );
					dvGridVert[1] = he3d_CVector( x + deltax, y - deltay, z + deltaz );
					dvGridVert[2] = he3d_CVector( x + deltax, y - deltay, z );
					dvGridVert[3] = he3d_CVector( x + deltax, y, z + deltaz );
										
					planeIndex = j*m_dwGridDensity + k;
					
					// te indexy leca z poprzedniego oczka				
					wGridIndices[2]  = wGridIndices[0];					
					wGridIndices[6]  = wGridIndices[4];					
					wGridIndices[11] = wGridIndices[8];					
					wGridIndices[10] = wGridIndices[9];					

					// z poprzedniego rzedu
					wGridIndices[4]  = m_pRow[k];
					wGridIndices[5]  = m_pRow[k + m_dwGridDensity];
					wGridIndices[7]  = m_pRow[k + (m_dwGridDensity<<1)];

					// z poprzedniej plaszczyny
					wGridIndices[8]  = m_pPlane[planeIndex + dwPlaneCount];					
					wGridIndices[3]	 = m_pPlane[planeIndex];
					m_pRow[k + (m_dwGridDensity<<1)] = m_pPlane[planeIndex];


					// nowe vertexy... dodajemy je do tablicy i wrzycamy odpowiednie indexy
					// do tablic z poprzedniego rzedu i plaszczyzny
					if( edge&1 ) 			
					{								
						pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[0], dvGridVert[1], fGridVal[0], fGridVal[1] );						
						
						m_pRow[k] = dwVertsCount;
						wGridIndices[0]	= dwVertsCount;

						dwVertsCount++;
					}

					if( edge&512)      
					{	
						pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[1], dvGridVert[3], fGridVal[1], fGridVal[5] );						

						m_pPlane[planeIndex + dwPlaneCount] = dwVertsCount;
						wGridIndices[9]	= dwVertsCount;

						dwVertsCount++;
					}

					if( edge&2 ) 					
					{					
						pVertTable[dwVertsCount].v = Interpolate( hullValue, dvGridVert[1], dvGridVert[2], fGridVal[1], fGridVal[2] );

						m_pRow[k + m_dwGridDensity] = dwVertsCount;
						m_pPlane[planeIndex] = dwVertsCount;
						wGridIndices[1] = dwVertsCount;

						dwVertsCount++;
					}
					
					for( l = 0 ; triTable[index][l] != -1 ; l++ )					
						pInd[dwIndsCount++] = wGridIndices[triTable[index][l]];
				}
			}
		}
	}
	
	ib->Unlock();

	dwFacesCount = dwIndsCount/3;

	ColorAndNormal( dwVertsCount, pVertTable, dwSign );

	vb->Unlock();
}

VOID CBlobSystem::ColorAndNormal( DWORD dwVertsCount, BLOBVERTEX* pVertTable, DWORD dwSign )
{
	he3d_CVector		n;
	he3d_CVector		vec;

	FLOAT				hullValue;		

	FLOAT				r, g, b;
	
	CBlob3D*			pBlob;	
	CNode*				node;

	FLOAT				tmp;
	DWORD				color;	

	if( dwSign )
		hullValue = -NEGBORDERVALUE;
	else
		hullValue = POSBORDERVALUE;

	hullValue = 1.0f/hullValue;	

	for( DWORD i = 0 ; i < dwVertsCount ; i++ )
	{
		n = he3d_CVector( 0, 0, 0 );		
		r = 0; g = 0; b = 0;

		for( node = root ; node ; node = node->next )
		{
			pBlob = node->blob;

			if( pBlob->m_dwSign == dwSign )
			{
				tmp	= pBlob->BlobFunction( pVertTable[i].v )*hullValue;						
				vec = pVertTable[i].v - pBlob->m_vPos;												

				n = n + tmp*Normalize( vec );				
				
				color	= pBlob->m_dwColor;					

				r += (FLOAT)( ( color>>16 )&0xff )*tmp;
				g += (FLOAT)( ( color>>8 )&0xff )*tmp;
				b += (FLOAT)( ( color&0xff ) )*tmp;				
			}
		}
		
		pVertTable[i].n = Normalize( n );		
		
		if( dwSign )
		{
			r = -r; b = -b; g = -g;
		}

		r = CLAMP(r); g = CLAMP(g); b = CLAMP(b);		

		pVertTable[i].c = ((DWORD)r<<16) | ((DWORD)g<<8) | (DWORD)b;		
	}
}

VOID CBlobSystem::ResizeToFit()
{
	if( !m_dwBlobCount )
		return;

	CBlob3D*		blob;
	CNode*			node = root;	

	blob = node->blob;

	FLOAT			minx, miny, minz;
	FLOAT			maxx, maxy, maxz;	
	
	FLOAT			r	= blob->m_fRadius;
	he3d_CVector	pos = blob->m_vPos;	

	minx = pos.x - r; maxx = pos.x + r;
	miny = pos.y - r; maxy = pos.y + r;
	minz = pos.z - r; maxz = pos.z + r;

	for( node = node->next ; node ; node = node->next )
	{	
		blob	= node->blob;

		r		= blob->m_fRadius;
		pos		= blob->m_vPos;

		if( pos.x - r < minx )
			minx = pos.x - r;

		if( pos.x + r > maxx )
			maxx = pos.x + r;

		if( pos.y - r < miny )
			miny = pos.y - r;

		if( pos.y + r > maxy )
			maxy = pos.y + r;

		if( pos.z - r < minz )
			minz = pos.z - r;

		if( pos.z + r > maxz )
			maxz = pos.z + r;	
	}

	m_vSize.x = maxx - minx;
	m_vSize.y = maxy - miny;
	m_vSize.z = maxz - minz;

	m_vCenter.x = minx + 0.5f*m_vSize.x;
	m_vCenter.y = miny + 0.5f*m_vSize.y;
	m_vCenter.z = minz + 0.5f*m_vSize.z;
}

VOID CBlobSystem::Render( PDIRECT3DDEVICE8 pDevice, DWORD dwSign )
{
	if( !pDevice )
		return;		

	Triangulate( m_pvbVertices, m_pibIndices, m_dwVerticesCount, m_dwFacesCount, dwSign );

	pDevice->SetStreamSource( 0, m_pvbVertices, sizeof(BLOBVERTEX) );
	pDevice->SetIndices( m_pibIndices, 0 );
	pDevice->SetVertexShader( BLOBFVF );	
	pDevice->DrawIndexedPrimitive( D3DPT_TRIANGLELIST, 0, m_dwVerticesCount, 0, m_dwFacesCount );	
	pDevice->SetStreamSource( 0, NULL, 0 );
	pDevice->SetIndices( NULL, 0 );
}

#pragma warning( default : 4244 )