/*
 * JUDAS V2.0 WAV handling
 */

#include <stdlib.h>
#include <stdio.h>
#include <mem.h>
#include "judas.h"

SAMPLE *judas_loadwav(char *name);

typedef struct
{
        char rifftext[4];
        unsigned totallength;
        char wavetext[4];
        char formattext[4];
        unsigned formatlength;
        unsigned short format;
        unsigned short channels;
        unsigned freq;
        unsigned avgbytes;
        unsigned short blockalign;
        unsigned short bits;
        char datatext[4];
        unsigned datalength;
} WAV_HEADER;

SAMPLE *judas_loadwav(char *name)
{
        int length;
        int reallength;
        int handle;
        SAMPLE *smp;
        WAV_HEADER header;

        /* Don't waste memory if Nosound */
        judas_error = JUDAS_OK;
        if (judas_device == DEV_NOSOUND)
        {
                return &fakesample;
        }

        /*
         * Try to open
         */
        judas_error = JUDAS_OPEN_ERROR;
        handle = judas_open(name);
        if (handle == -1) return NULL;

        /*
         * Read identification
         */
        judas_error = JUDAS_READ_ERROR;
        if (judas_read(handle, &header, 12) != 12)
        {
                judas_close(handle);
                return NULL;
        }
        judas_error = JUDAS_WRONG_FORMAT;
        if (memcmp("RIFF", header.rifftext, 4))
        {
                judas_close(handle);
                return NULL;
        }
        if (memcmp("WAVE", header.wavetext, 4))
        {
                judas_close(handle);
                return NULL;
        }
        /*
         * Search for the FORMAT chunk
         */
        for (;;)
        {
                judas_error = JUDAS_READ_ERROR;
                if (judas_read(handle, &header.formattext, 8) != 8)
                {
                        judas_close(handle);
                        return NULL;
                }
                if (!memcmp("fmt ", &header.formattext, 4)) break;
                if (judas_seek(handle, header.formatlength, SEEK_CUR) == -1)
                {
                        judas_close(handle);
                        return NULL;
                }
        }
        /*
         * Read in the FORMAT chunk
         */
        if (judas_read(handle, &header.format, 16) != 16)
        {
                judas_close(handle);
                return NULL;
        }
        /*
         * Skip data if the format chunk was bigger than what we use
         */
        if (judas_seek(handle, header.formatlength - 16, SEEK_CUR) == -1)
        {
                judas_close(handle);
                return NULL;
        }
        /*
         * Check for correct format
         */
        judas_error = JUDAS_WRONG_FORMAT;
        if (header.format != 1)
        {
                judas_close(handle);
                return NULL;
        }
        /*
         * Search for the DATA chunk
         */
        for (;;)
        {
                judas_error = JUDAS_READ_ERROR;
                if (judas_read(handle, &header.datatext, 8) != 8)
                {
                        judas_close(handle);
                        return NULL;
                }
                if (!memcmp("data", &header.datatext, 4)) break;
                if (judas_seek(handle, header.datalength, SEEK_CUR) == -1)
                {
                        judas_close(handle);
                        return NULL;
                }
        }
        /*
         * Allocate sample, load audio data, do processing (unsigned->signed,
         * stereo->mono)
         */
        length = header.datalength;
        reallength = length;
        if (header.channels == 2) reallength >>= 1;
        smp = judas_allocsample(reallength);
        if (!smp)
        {
                judas_close(handle);
                return NULL;
        }
        if (header.channels == 2)
        {
                if (header.bits == 16)
                {
                        unsigned count = length >> 2;
                        short *buffer;
                        short *src;
                        short *dest;

                        judas_error = JUDAS_OUT_OF_MEMORY;
                        buffer = malloc(length);
                        if (!buffer)
                        {
                                judas_freesample(smp);
                                judas_close(handle);
                                return NULL;
                        }
                        judas_error = JUDAS_READ_ERROR;
                        if (judas_read(handle, buffer, length) != length)
                        {
                                free(buffer);
                                judas_freesample(smp);
                                judas_close(handle);
                                return NULL;
                        }
                        src = buffer;
                        dest = (short *)smp->start;
                        while (count--)
                        {
                                int average = (src[0] + src[1]) / 2;
                                *dest = average;
                                src += 2;
                                dest++;
                        }
                        free(buffer);
                        smp->repeat = smp->start;
                        smp->end = smp->start + reallength;
                        smp->voicemode = VM_ON | VM_16BIT;
                }
                else
                {
                        unsigned count = length >> 1;
                        unsigned char *buffer;
                        unsigned char *src;
                        signed char *dest;

                        judas_error = JUDAS_OUT_OF_MEMORY;
                        buffer = malloc(length);
                        if (!buffer)
                        {
                                judas_freesample(smp);
                                judas_close(handle);
                                return NULL;
                        }
                        judas_error = JUDAS_READ_ERROR;
                        if (judas_read(handle, buffer, length) != length)
                        {
                                free(buffer);
                                judas_freesample(smp);
                                judas_close(handle);
                                return NULL;
                        }
                        src = buffer;
                        dest = smp->start;
                        while (count--)
                        {
                                int average = (src[0] + src[1] - 0x100) / 2;
                                *dest = average;
                                src += 2;
                                dest++;
                        }
                        free(buffer);
                        smp->repeat = smp->start;
                        smp->end = smp->start + reallength;
                        smp->voicemode = VM_ON;
                }
        }
        else
        {
                if (header.bits == 16)
                {
                        judas_error = JUDAS_READ_ERROR;
                        if (judas_read(handle, smp->start, length) != length)
                        {
                                judas_freesample(smp);
                                judas_close(handle);
                                return NULL;
                        }
                        smp->repeat = smp->start;
                        smp->end = smp->start + length;
                        smp->voicemode = VM_ON | VM_16BIT;
                }
                else
                {
                        unsigned count = length;
                        char *src = smp->start;

                        judas_error = JUDAS_READ_ERROR;
                        if (judas_read(handle, smp->start, length) != length)
                        {
                                judas_freesample(smp);
                                judas_close(handle);
                                return NULL;
                        }
                        while (count--)
                        {
                                *src += 0x80;
                                src++;
                        }
                        smp->repeat = smp->start;
                        smp->end = smp->start + length;
                        smp->voicemode = VM_ON;
                }
        }
        judas_ipcorrect(smp);
        judas_error = JUDAS_OK;
        judas_close(handle);
        return smp;
}
