//===============================================================================================
// FMOD.EXE
// Copyright (c), Firelight Multimedia, 1999-2000.
//
// This example really demonstrates the FMOD music system at work, and is a standalone 
// mod/wav/mp3 player!
// It displays a lot of information and uses the DSP engine as well to provide DSP effects under
// MMX only.
//===============================================================================================

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <stdio.h>
#include <time.h>

#include "../../api/inc/fmod.h"
#include "../../api/inc/fmod_errors.h"	// optional

#include "resource.h"
#include "sdriver.h"
#include "lowpass.h"

// ===============================================================================================
// DEFINITIONS
// ===============================================================================================
#define NAME 			"FSOUND TESTBED"
#define TITLE 			"FMOD"
#define NUMCHANNELS		128
#define MAXSONGS		512
#define PREVERB_NUMTAPS	3
#define REVERB_NUMTAPS	7

typedef struct
{
	FSOUND_DSPUNIT	*Unit;
	char			*historybuff;		// storage space for tap history
	char			*workarea;			// a place to hold 1 buffer worth of data (for preverb)
	int				delayms;			// delay of p/reverb tab in milliseconds
	int				volume;				// volume of p/reverb tab
	int				pan;				// pan of p/reverb tab
	int				historyoffset;		// running offset into history buffer
	int				historylen;			// size of history buffer in SAMPLES
} REVERBTAP;


typedef struct
{
	FMUSIC_MODULE	*mod;
	FSOUND_STREAM	*stream;
	int				channel;
} SONGTYPE;

// ===============================================================================================
// GLOBALS
// ===============================================================================================

HWND			mainhwnd;
HWND			cdhwnd;

UINT			timerid=0;
UINT			cdtimerid=0;
SONGTYPE		song[MAXSONGS+1];

int				outputfreq = 44100;				// default output freq
int				playlistsong=0;
int				DSP_Ready = FALSE;
float			scalex = 1.0f, scaley = 1.0f;

// lowpass stuff
FSOUND_DSPUNIT	*DSP_LowPassUnit;
signed char		*DSP_LowPassBuffer;
float			DSP_LowPassCutoffFrequency = 5000.0f;
float			DSP_LowPassResonance = 1.0f;

// echo stuff
#define MAXECHOLEN (500 * outputfreq / 1000)
FSOUND_DSPUNIT *DSP_EchoUnit = NULL;
signed char	*	DSP_EchoBuffer = NULL;
int				DSP_EchoOffset=0, DSP_EchoLen=0;

// preverb stuff
REVERBTAP		DSP_PreverbTap[PREVERB_NUMTAPS];

// reverb stuff
REVERBTAP		DSP_ReverbTap[REVERB_NUMTAPS];


//===============================================================================================
// SETTINGS 
//===============================================================================================

int setting_xpos=0;
int setting_ypos=0;
int setting_cdxpos=0;
int setting_cdypos=0;
int setting_cdmode=FALSE;

int setting_output=0;
int setting_driver=0;
int setting_mixer=0;
int setting_outputrate=44100;

//===============================================================================================
// PROTOTYPES 
//===============================================================================================

void CloseDown();

// ===============================================================================================
// THE CODE!
// ===============================================================================================

void LoadSettings()
{
	HKEY key;
	DWORD result,size=4;

	result = RegOpenKeyEx(HKEY_LOCAL_MACHINE, "Software\\FMOD\\Settings", 0, KEY_QUERY_VALUE , &key);

	setting_output = FSOUND_OUTPUT_DSOUND;
	setting_driver = 0;
	setting_mixer = FSOUND_MIXER_QUALITY_AUTODETECT;
	setting_outputrate = 44100;

	if (result != ERROR_SUCCESS) 
		return;

	RegQueryValueEx(key, "xpos", 0, NULL, (LPBYTE)&setting_xpos, &size);
	RegQueryValueEx(key, "ypos", 0, NULL, (LPBYTE)&setting_ypos, &size);
	RegQueryValueEx(key, "cdxpos", 0, NULL, (LPBYTE)&setting_cdxpos, &size);
	RegQueryValueEx(key, "cdypos", 0, NULL, (LPBYTE)&setting_cdypos, &size);
	RegQueryValueEx(key, "cdmode", 0, NULL, (LPBYTE)&setting_cdmode, &size);

	RegQueryValueEx(key, "output", 0, NULL, (LPBYTE)&setting_output, &size);
	RegQueryValueEx(key, "driver", 0, NULL, (LPBYTE)&setting_driver, &size);
	RegQueryValueEx(key, "mixer",  0, NULL, (LPBYTE)&setting_mixer, &size);
	RegQueryValueEx(key, "outputrate",  0, NULL, (LPBYTE)&setting_outputrate, &size);

	// it is not possible to select nosound from dialog box, so it must be an uninitialized
	// registry key.. anyway, however it was 0, set it to directsound if it is.
	if (!setting_output) 
		setting_output = FSOUND_OUTPUT_DSOUND;
	if (setting_outputrate < 4000) 
		setting_outputrate = 44100;

	RegCloseKey(key);
}

void SaveSettings()
{
	HKEY key;
	DWORD result;

	setting_output	= FSOUND_GetOutput();
	setting_driver	= FSOUND_GetDriver();
	setting_mixer	= FSOUND_GetMixer();
	setting_outputrate = FSOUND_GetOutputRate();

	RegCreateKeyEx(HKEY_LOCAL_MACHINE, "Software\\FMOD\\Settings", 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key, &result);
	RegSetValueEx(key, "xpos", 0, REG_DWORD, (LPBYTE)&setting_xpos, 4);
	RegSetValueEx(key, "ypos", 0, REG_DWORD, (LPBYTE)&setting_ypos, 4);
	RegSetValueEx(key, "cdxpos", 0, REG_DWORD, (LPBYTE)&setting_cdxpos, 4);
	RegSetValueEx(key, "cdypos", 0, REG_DWORD, (LPBYTE)&setting_cdypos, 4);
	RegSetValueEx(key, "cdmode", 0, REG_DWORD, (LPBYTE)&setting_cdmode, 4);

	RegSetValueEx(key, "output", 0, REG_DWORD, (LPBYTE)&setting_output, 4);
	RegSetValueEx(key, "driver", 0, REG_DWORD, (LPBYTE)&setting_driver, 4);
	RegSetValueEx(key, "mixer",  0, REG_DWORD, (LPBYTE)&setting_mixer, 4);
	RegSetValueEx(key, "outputrate",  0, REG_DWORD, (LPBYTE)&setting_outputrate, 4);
	RegCloseKey(key);
}





/*
[
	[DESCRIPTION]
	A slow and quite unscientific 4th order low pass filter, which leaves the source mix 
	buffer in tact, and returns a pointer to the new 'dirty' buffer.

	[PARAMETERS]
	'originalbuffer'	Pointer to the original mixbuffer, not any buffers passed down 
						through the dsp chain.  They are in newbuffer.
	'newbuffer'			Pointer to buffer passed from previous DSP unit.
	'length'			Length in SAMPLES of buffer being passed.
	'param'				User parameter.  In this case it is a pointer to DSP_LowPassBuffer.
 
	[RETURN_VALUE]
	A pointer to the new modified lowpass buffer, leaving the original buffer untouched.

	[REMARKS]
	Leaving the original source data untouched means we still have clean source data to 
	work with using other chained filters.
]
*/
void *DSP_LowPassCallback(void *originalbuffer, void *newbuffer, int length, int param)
{
	int				count;
	signed short	*srcleft, *srcright, *destleft, *destright;
	int				mixertype = FSOUND_GetMixer();
	
	// must be 16bit stereo integer buffer.. sorry blendmode (32bit) and fpu (32bit float) dont support this.
	if (mixertype == FSOUND_MIXER_BLENDMODE || 
		mixertype == FSOUND_MIXER_QUALITY_FPU)
		return newbuffer;

	srcleft   = ((signed short *)newbuffer);
	srcright  = ((signed short *)newbuffer)+1;
	destleft  = ((signed short *)param);
	destright = ((signed short *)param)+1;

	length <<= 1;	// *2 for stereo (number of 16 bit samples)


	for (count=0; count<length; count+=2)
	{
		signed int l, r;

		// this isnt correct lowpassing l and r with the same filter (ie it should have one
		// filter per channel) but its a quick hack and sounds ok for now.
		l = (signed int)LowPass_Filter((float)srcleft[count]);
		r = (signed int)LowPass_Filter((float)srcright[count]);

		if      (l < -32768) l = -32768;
		else if (l >  32767) l =  32767;
		if      (r < -32768) r = -32768;
		else if (r >  32767) r =  32767;
		
		destleft[count] = (signed short)l;
		destright[count] = (signed short)r;
	}

	// data has been copied into new buffer, old buffer is still in tact (clean) for another
	// filter
	return (void *)param;
}


/*
[
	[DESCRIPTION]
	Callback to mix in one reverb tap.  It copies the buffer into its own history buffer also.

	[PARAMETERS]
	'originalbuffer'	Pointer to the original mixbuffer, not any buffers passed down 
						through the dsp chain.  They are in newbuffer.
	'newbuffer'			Pointer to buffer passed from previous DSP unit.
	'length'			Length in SAMPLES of buffer being passed.
	'param'				User parameter.  In this case it is a pointer to DSP_LowPassBuffer.
 
	[RETURN_VALUE]
	a pointer to the buffer that was passed in, with a tap mixed into it.

	[REMARKS]
]
*/
void *DSP_ReverbCallback(void *originalbuffer, void *newbuffer, int length, int param)
{
	int		mixertype = FSOUND_GetMixer();
	REVERBTAP	*tap = (REVERBTAP *)param;

	// must be 16bit stereo integer buffer.. sorry blendmode (32bit) and fpu (32bit float) dont support this.
	if (mixertype == FSOUND_MIXER_BLENDMODE || 
		mixertype == FSOUND_MIXER_QUALITY_FPU)
		return newbuffer;

#define FEEDBACK 1	// this determines whether to use already pre-reverbed data for the tap, or the original clean buffer

	// reverb history buffer is a ringbuffer.  If the length makes the copy wrap, then split the copy 
	// into end part, and start part.. 
	if (tap->historyoffset + length > tap->historylen)
	{
		int taillen = tap->historylen - tap->historyoffset;
		int startlen = length - taillen;

		// mix a scaled version of history buffer into output
		FSOUND_DSP_MixBuffers(newbuffer, tap->historybuff + (tap->historyoffset << 2), taillen, outputfreq, tap->volume, tap->pan, FSOUND_STEREO | FSOUND_16BITS);
		FSOUND_DSP_MixBuffers((char *)newbuffer+((tap->historylen - tap->historyoffset) << 2), tap->historybuff, startlen, outputfreq, tap->volume, tap->pan, FSOUND_STEREO | FSOUND_16BITS);

		// now copy input into reverb/history buffer 
#ifdef FEEDBACK
		memcpy(tap->historybuff + (tap->historyoffset << 2), newbuffer, taillen<<2);
		memcpy(tap->historybuff, (char *)newbuffer + ((tap->historylen - tap->historyoffset) << 2), startlen << 2);
#else
		memcpy(tap->historybuff + (tap->historyoffset << 2), originalbuffer, taillen<<2);
		memcpy(tap->historybuff, (char *)originalbuffer + ((tap->historylen - tap->historyoffset) << 2), startlen << 2);
#endif

	}
	// no wrapping reverb buffer, just write dest
	else
	{
		// mix a scaled version of history buffer into output
		FSOUND_DSP_MixBuffers(newbuffer, tap->historybuff + (tap->historyoffset << 2), length, outputfreq, tap->volume, tap->pan, FSOUND_STEREO | FSOUND_16BITS);

		// now copy input into reverb/history buffer 
#ifdef FEEDBACK
		memcpy(tap->historybuff + (tap->historyoffset << 2), newbuffer, length << 2);
#else
		memcpy(tap->historybuff + (tap->historyoffset << 2), originalbuffer, length << 2);
#endif
	}


	tap->historyoffset += length;
	if (tap->historyoffset >= tap->historylen) 
		tap->historyoffset -= tap->historylen;

	// reverb history has been mixed into new buffer, so return it.
	return newbuffer;
}


/*
[
	[DESCRIPTION]
	Simple echo callback.  It copies the data into its own history buffer, then copies the data back in.

	[PARAMETERS]
	'originalbuffer'	Pointer to the original mixbuffer, not any buffers passed down 
						through the dsp chain.  They are in newbuffer.
	'newbuffer'			Pointer to buffer passed from previous DSP unit.
	'length'			Length in SAMPLES of buffer being passed.
	'param'				User parameter.  In this case it is a pointer to DSP_EchoBuffer.
 
	[RETURN_VALUE]
	A pointer to the new modified buffer.  The buffer passed in has been modified, and 
	DSP_EchoBuffer has just been filled with a copy of the original information to keep
	an echo history for later.

	[REMARKS]
	All the <<2 stuff is to convert samples to bytes, as all offsets and lengths are based 
	on samples not bytes.  For mmx the output size is 4 bytes per sample.

	[SEE_ALSO]
	DSP_LowPassCallback
]
*/
void *DSP_EchoCallback(void *originalbuffer, void *newbuffer, int length, int param)
{
	int mixertype = FSOUND_GetMixer();
	char *echobuff = (char *)param;
	
	// must be 16bit stereo integer buffer.. sorry blendmode (32bit) and fpu (32bit float) dont support this.
	if (mixertype == FSOUND_MIXER_BLENDMODE || 
		mixertype == FSOUND_MIXER_QUALITY_FPU)
		return newbuffer;


	// echobuff is a ringbuffer that we copy the mixbuffer to.  
	// If the length of the write exceeds the end of the echo buffer,
	// then do the mix in 2 parts, the end part, and the start part.
	if (DSP_EchoOffset + length > DSP_EchoLen)
	{
		int taillen = DSP_EchoLen - DSP_EchoOffset;
		int startlen = length - taillen;
		
		// feedback history from echo buffer into mixbuffer
		FSOUND_DSP_MixBuffers(newbuffer, echobuff+(DSP_EchoOffset << 2), taillen,	  outputfreq, 128, FSOUND_STEREOPAN, FSOUND_STEREO | FSOUND_16BITS);
		FSOUND_DSP_MixBuffers((char *)newbuffer + (taillen << 2), echobuff, startlen, outputfreq, 128, FSOUND_STEREOPAN, FSOUND_STEREO | FSOUND_16BITS);

		// now copy result into echo buffer again for next time
		memcpy(echobuff + (DSP_EchoOffset << 2), newbuffer, taillen << 2);
		memcpy(echobuff, (char *)newbuffer + (taillen << 2), startlen << 2);
	}
	// no wrapping echo buffer write, just do a straight write 
	else
	{
		// feedback history from echo buffer into mixbuffer
		FSOUND_DSP_MixBuffers(newbuffer, echobuff + (DSP_EchoOffset << 2), length, outputfreq, 128, FSOUND_STEREOPAN, FSOUND_STEREO | FSOUND_16BITS);
		// now copy result into echo buffer again for next time
		memcpy(echobuff + (DSP_EchoOffset << 2), newbuffer, length << 2);
	}

	DSP_EchoOffset+=length;
	if (DSP_EchoOffset >= DSP_EchoLen) 
		DSP_EchoOffset -= DSP_EchoLen;

	// echo history has been mixed into new buffer, so return it.
	return newbuffer;
}


/*
[
	[DESCRIPTION]
	Similair to a reverb tap except the history copy and the tap mix are done back to front.

	[PARAMETERS]
	'originalbuffer'	Pointer to the original mixbuffer, not any buffers passed down 
						through the dsp chain.  They are in newbuffer.
	'newbuffer'			Pointer to buffer passed from previous DSP unit.
	'length'			Length in SAMPLES of buffer being passed.
	'param'				User parameter.  In this case it is a pointer to a REVERBTAP structure
 
	[RETURN_VALUE]
	A pointer to the new modified buffer.  

	[REMARKS]
	All the <<2 stuff is to convert samples to bytes, as all offsets and lengths are based 
	on samples not bytes.  For mmx the output size is 4 bytes per sample.

	[SEE_ALSO]
	DSP_LowPassCallback
]
*/
void *DSP_PreverbCallback(void *originalbuffer, void *newbuffer, int length, int param)
{
	int		mixertype = FSOUND_GetMixer();
	REVERBTAP	*tap = (REVERBTAP *)param;

	// must be 16bit stereo integer buffer.. sorry blendmode (32bit) and fpu (32bit float) dont support this.
	if (mixertype == FSOUND_MIXER_BLENDMODE || 
		mixertype == FSOUND_MIXER_QUALITY_FPU)
		return newbuffer;


	// preverbbuff is a ringbuffer.  If the length makes the copy wrap, then split the copy 
	// into end part, and start part
	if (tap->historyoffset + length > tap->historylen)
	{
		int taillen = tap->historylen - tap->historyoffset;
		int startlen = length - taillen;		// whatever is left

		// get a clean version of the preverb buffer (should be an unscaled history of the mixbuffer)
		memcpy(tap->workarea, tap->historybuff + (tap->historyoffset << 2), (tap->historylen - tap->historyoffset)<<2);
		memcpy(tap->workarea + ((tap->historylen - tap->historyoffset) << 2), tap->historybuff, (length - (tap->historylen - tap->historyoffset)) << 2);

		// now copy input into preverb/history buffer 
		memcpy(tap->historybuff + (tap->historyoffset << 2), newbuffer, (tap->historylen - tap->historyoffset)<<2);
		memcpy(tap->historybuff, (signed char *)newbuffer + ((tap->historylen - tap->historyoffset) << 2), (length - (tap->historylen - tap->historyoffset)) << 2);

	}
	// no wrapping preverb buffer, just write dest
	else
	{
		// get a clean version of the preverb buffer (should be an unscaled history of the mixbuffer)
		memcpy(tap->workarea, tap->historybuff + (tap->historyoffset << 2), length << 2);

		// now copy input into preverb/history buffer 
		memcpy(tap->historybuff + (tap->historyoffset << 2), newbuffer, length << 2);
	}

	// now mix a scaled input into this 
	FSOUND_DSP_MixBuffers(tap->workarea, newbuffer, length, outputfreq, tap->volume, tap->pan, FSOUND_STEREO | FSOUND_16BITS);

	tap->historyoffset += length;
	if (tap->historyoffset >= tap->historylen) 
		tap->historyoffset -= tap->historylen;

	// preverb history has been mixed into new buffer, so return it.
	return tap->workarea;
}


/*
FMUSIC_MODULE *recsong;

void PlaySongResource()
{
	HRSRC rec;
	HGLOBAL handle;
	void *data;
	int length;

	rec = FindResource(GetModuleHandle(NULL), MAKEINTRESOURCE(IDR_MODULE1), RT_RCDATA);
	handle = LoadResource(NULL, rec);
	
	data = LockResource(handle);
	length = SizeofResource(NULL, rec);

	recsong = FMUSIC_LoadSongMemory(data, length);

	FMUSIC_PlaySong(recsong);
}

void StopSongResource()
{
	FMUSIC_StopSong(recsong);
	FMUSIC_FreeSong(recsong);
}
*/

/*
[
	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
]
*/
void LoadSong(HWND LB_Parent_Hwnd, HWND LB_Hwnd) 
{
	OPENFILENAME	ofn;					// common dialog box structure   
	char			szDirName[MAX_PATH];    // directory string             
	char			szFile[2048];			// filename string               
	char			szFileTitle[2048];      // filename string               
	FMUSIC_MODULE	*mod = NULL;
	FSOUND_STREAM	*stream = NULL;
	int				currsong=0;

	// hinst = (HINSTANCE)GetWindowLong(hwnd, GWL_HINSTANCE);

	// Obtain the system directory name and store it in szDirName. 
	GetSystemDirectory(szDirName, sizeof(szDirName)); 

	// Place the terminating null character in the szFile.
	szFile[0] = '\0'; 

	// Set the members of the OPENFILENAME structure.
	memset(&ofn, 0, sizeof(ofn));
	ofn.lStructSize = sizeof(OPENFILENAME); 
	ofn.hwndOwner = LB_Parent_Hwnd; 
	ofn.lpstrTitle = "Open\0";
	ofn.lpstrFilter = "All song Types\0*.MOD;*.S3M;*.XM;*.IT;*.MID;*.RMI;*.SGT;*.WAV;*.MP2;*.MP3;*.OGG;*.WMA;*.ASF\0Microsoft WAV (*.WAV)\0*.WAV\0MP2/MP3 (*.MP3 *.MP2)\0*.MP2;*.MP3\0Ogg Vorbis (*.OGG)\0*.OGG\0Windows Media Format (*.WMA *.ASF)\0*.WMA;*.ASF\0MIDI / DirectMusic Files (*.MID,*.RMI,*.SGT)\0*.MID;*.RMI;*.SGT\0Impulse Tracker (*.IT)\0*.IT\0FastTracker2 (*.XM)\0*.XM\0ScreamTracker 3 (*.S3M)\0*.S3M\0Protracker/FastTracker (*.MOD)\0*.MOD\0All files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter = NULL;
	ofn.nFilterIndex = 1; 
	ofn.lpstrFile = szFile; 
	ofn.nMaxFile = sizeof(szFile); 
	ofn.lpstrFileTitle = szFileTitle;
	ofn.nMaxFileTitle = sizeof(szFileTitle);
	ofn.lpstrInitialDir = ".\0";
	ofn.Flags = OFN_EXPLORER | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_ALLOWMULTISELECT; 

	// Display the Open dialog box.
	if (GetOpenFileName(&ofn)) 
	{ 
		char *path = ofn.lpstrFile;
		char *fname = ofn.lpstrFile;
		char name[256];

		// single file
		if (ofn.nFileOffset)
		{
			// skip to the first filename
			fname +=ofn.nFileOffset;
			*(fname-1) = 0;		// clear the backslash before the filename so the loop below handles it
		}
		// multiple files
		else
		{
			while (*fname++);	// search to the first filename
		}

		do 
		{
			char s[256];

			// put path in name
			strcpy(name, path);
			strcat(name, "\\");
			strcat(name, fname);

			// Open the file.
			stream = NULL;
			mod = FMUSIC_LoadSong(name);
			if (!mod)
				stream = FSOUND_Stream_OpenFile(name, FSOUND_NORMAL /*| FSOUND_LOOP_NORMAL*/ | FSOUND_2D, 0);

			if (!mod && !stream)
			{
				MessageBox(LB_Parent_Hwnd, FMOD_ErrorString(FSOUND_GetError()), "Loading a song", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
				
				while (*fname++);
				
				continue;
			}

			if (mod)
			{
				if (FMUSIC_GetType(mod) != FMUSIC_TYPE_IT)		// IT has its own master volume setting
					FMUSIC_SetMasterVolume(mod, 128);
				if (FMUSIC_GetType(mod) == FMUSIC_TYPE_MOD || FMUSIC_GetType(mod) == FMUSIC_TYPE_S3M)
					FMUSIC_SetPanSeperation(mod, 0.85f);	// 15% crossover
			}

			currsong = SendMessage(LB_Hwnd, LB_GETCOUNT, 0, 0);
			if (currsong >= MAXSONGS) 
			{
				MessageBox(LB_Parent_Hwnd, "Error.  Unable to fit any more songs in.. please restart app", "Loading a song", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
				break;
			}
			
			if (mod)
			{
				song[currsong].mod = mod;
				song[currsong].stream = NULL;
			}
			else
			{
				song[currsong].stream = stream;
				song[currsong].mod = NULL;
			}
			
			strcpy(s, fname);

			SendMessage(LB_Hwnd, LB_ADDSTRING, 0, (LPARAM) (LPCTSTR)s);
			UpdateWindow(LB_Hwnd);
			SendMessage(LB_Hwnd, LB_SETCURSEL, currsong, 0);

			while (*fname++);

		} while (*fname);
	}
}



/*
[
	[DESCRIPTION]
	Create DSP units and reverb buffers etc

	[PARAMETERS]
	void
 
	[RETURN_VALUE]
	void

	[REMARKS]

	[SEE_ALSO]
	CloseDSP
]
*/
void InitDSP()
{
	DSP_Ready = FALSE;

	DSP_LowPassBuffer = calloc(FSOUND_DSP_GetBufferLength()+256, 4);	

	DSP_EchoLen		= MAXECHOLEN;				// 500ms
	DSP_EchoBuffer  = calloc(DSP_EchoLen, 4);	
	DSP_EchoOffset	= 0;

	LowPass_Init();
	LowPass_Update(DSP_LowPassResonance, DSP_LowPassCutoffFrequency, outputfreq);

	DSP_LowPassUnit = FSOUND_DSP_Create(&DSP_LowPassCallback,	FSOUND_DSP_DEFAULTPRIORITY_USER+1,	(int)DSP_LowPassBuffer);
	DSP_EchoUnit	= FSOUND_DSP_Create(&DSP_EchoCallback,		FSOUND_DSP_DEFAULTPRIORITY_USER+2,	(int)DSP_EchoBuffer);

	// ====================================================================================================================
	// PREVERB SETUP
	// ====================================================================================================================
	{
		int delay[PREVERB_NUMTAPS]	= { 57, 97, 163 };		// // prime numbers! dont go lower than 20!  it will be smaller than the DSP bufferlen!! (a check is done below for this)
		int volume[PREVERB_NUMTAPS] = { 128, 78, 46 };
		int pan[PREVERB_NUMTAPS]	= { 128-24, 128+24, 128 };
		int count;

		for (count=0; count< PREVERB_NUMTAPS; count++)
		{
			DSP_PreverbTap[count].delayms		= delay[count];	
			DSP_PreverbTap[count].volume		= volume[count];
			DSP_PreverbTap[count].pan			= pan[count];
			DSP_PreverbTap[count].historyoffset = 0;
			DSP_PreverbTap[count].historylen	= (DSP_PreverbTap[count].delayms * outputfreq / 1000);
			
			if (DSP_PreverbTap[count].historylen < FSOUND_DSP_GetBufferLength())
				DSP_PreverbTap[count].historylen = FSOUND_DSP_GetBufferLength();	// just in case our calc is not the same.

			DSP_PreverbTap[count].historybuff	= calloc(DSP_PreverbTap[count].historylen + 2048, 4);	// * 4 is for 16bit stereo (mmx only)
			DSP_PreverbTap[count].workarea		= calloc(FSOUND_DSP_GetBufferLength(), 4); 
			DSP_PreverbTap[count].Unit			= FSOUND_DSP_Create(&DSP_PreverbCallback, FSOUND_DSP_DEFAULTPRIORITY_USER+count,	(int)&DSP_PreverbTap[count]);
		}
	}

	// ====================================================================================================================
	// REVERB SETUP
	// ====================================================================================================================
	{
		// something to fiddle with..		
		int delay[REVERB_NUMTAPS]	= { 131, 149, 173, 211, 281, 401, 457};	// prime numbers!
		int volume[REVERB_NUMTAPS]	= { 120, 100,  95,  90,  80,  60,  50};
		int pan[REVERB_NUMTAPS]		= { 100, 128, 128, 152, 128, 100, 152};
		int count;

		for (count=0; count< REVERB_NUMTAPS; count++)
		{
			DSP_ReverbTap[count].delayms		= delay[count];	
			DSP_ReverbTap[count].volume			= volume[count];
			DSP_ReverbTap[count].pan			= pan[count];
			DSP_ReverbTap[count].historyoffset	= 0;
			DSP_ReverbTap[count].historylen		= (DSP_ReverbTap[count].delayms * outputfreq / 1000);
			if (DSP_ReverbTap[count].historylen < FSOUND_DSP_GetBufferLength())
				DSP_ReverbTap[count].historylen = FSOUND_DSP_GetBufferLength();	// just in case our calc is not the same.

			DSP_ReverbTap[count].historybuff	= calloc(DSP_ReverbTap[count].historylen, 4);	// * 4 is for 16bit stereo (mmx only)
			DSP_ReverbTap[count].workarea		= NULL;
			DSP_ReverbTap[count].Unit			= FSOUND_DSP_Create(&DSP_ReverbCallback, FSOUND_DSP_DEFAULTPRIORITY_USER+20+(count*2),	(int)&DSP_ReverbTap[count]);
		}
	}

	DSP_Ready = TRUE;
}


/*
[
	[DESCRIPTION]
	Remove all DSP units and reverb buffers etc.

	[PARAMETERS]
	void
 
	[RETURN_VALUE]
	void

	[REMARKS]

	[SEE_ALSO]
	InitDSP
]
*/
void CloseDSP()
{
	int count;

	DSP_Ready = FALSE;

	if (DSP_LowPassUnit)
		FSOUND_DSP_Free(DSP_LowPassUnit);
	DSP_LowPassUnit = NULL;

	if (DSP_EchoUnit)
		FSOUND_DSP_Free(DSP_EchoUnit);
	DSP_EchoUnit = NULL;

	for (count=0; count<PREVERB_NUMTAPS; count++)
	{
		if (DSP_PreverbTap[count].Unit)
			FSOUND_DSP_Free(DSP_PreverbTap[count].Unit);
		DSP_PreverbTap[count].Unit = NULL;

		if (DSP_PreverbTap[count].historybuff)
			free(DSP_PreverbTap[count].historybuff);		
		DSP_PreverbTap[count].historybuff = NULL;

		if (DSP_PreverbTap[count].workarea)
			free(DSP_PreverbTap[count].workarea);
		DSP_PreverbTap[count].workarea = NULL;
	}

	for (count=0; count<REVERB_NUMTAPS; count++)
	{
		if (DSP_ReverbTap[count].Unit)
			FSOUND_DSP_Free(DSP_ReverbTap[count].Unit);	
		DSP_ReverbTap[count].Unit = NULL;

		if (DSP_ReverbTap[count].historybuff)
			free(DSP_ReverbTap[count].historybuff);
		DSP_ReverbTap[count].historybuff = NULL;

		if (DSP_ReverbTap[count].workarea)
			free(DSP_ReverbTap[count].workarea);
		DSP_ReverbTap[count].workarea = NULL;
	}

	if (DSP_LowPassBuffer)
		free(DSP_LowPassBuffer);
	DSP_LowPassBuffer = NULL;

	if (DSP_EchoBuffer)
		free(DSP_EchoBuffer);
	DSP_EchoBuffer = NULL;

	LowPass_Close();
}


/*
[
	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
]
*/
BOOL CALLBACK FMOD_DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg)
	{
		case WM_INITDIALOG:
		{
			int	count;
		
			SetFocus(hwnd);

			//===================================================================================
			// CALL DRIVER DIALOG BOX
			//===================================================================================
			#if 0
			if (!SoundDriver_Init(&outputfreq))
			{
				SendMessage(hwnd, WM_CLOSE, 0, 0);
	  			return TRUE;
			}
			#endif

			//===================================================================================
			// CONFIGURE STABILITY OF SOUND OUTPUT UNDER WINDOWS
			//===================================================================================

			FSOUND_SetBufferSize(200);

			//===================================================================================
			// INITIALIZE FSOUND
			//===================================================================================
			FSOUND_SetOutput(setting_output);
			FSOUND_SetDriver(setting_driver);
			FSOUND_SetMixer(setting_mixer);
			FSOUND_SetHWND(hwnd);

			outputfreq = setting_outputrate;

			if (!FSOUND_Init(outputfreq,NUMCHANNELS, 0))
			{
    			MessageBox(hwnd, FMOD_ErrorString(FSOUND_GetError()), "FSOUND", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
				SendMessage(hwnd, WM_CLOSE, 0, 0);
    			return TRUE;
			}



			//===================================================================================
			// SET UP A PAINT TIMER FOR INTERFACE
			//===================================================================================
			timerid = SetTimer(hwnd, 0, 10, 0L); 
			if (!timerid)
			{
    			MessageBox( NULL, "Too many timers in use", "Error", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
    			return FALSE;
			}

			//===================================================================================
			// INITIALIZE SONG LIST TO NULL
			//===================================================================================
			for (count=0; count<MAXSONGS+1; count++) 
			{
				song[count].mod = NULL;
				song[count].stream = NULL;
				song[count].channel = -1;
			}

			//===================================================================================
			// SET UP DSP UNITS
			//===================================================================================
			InitDSP();


			
			return TRUE;
		}

		case WM_CLOSE:
		{
			CloseDSP();

			KillTimer(hwnd, timerid);
			CloseDown();
			EndDialog(hwnd,0);
            PostQuitMessage(0);
			return TRUE;
		}
		case WM_HSCROLL:
		{
			HWND lbhwnd = GetDlgItem(hwnd,IDC_SONGLIST);
			int songid = SendMessage(lbhwnd, LB_GETCURSEL, 0, 0);

			if (songid != LB_ERR)
			{
				FMUSIC_MODULE *mod = song[songid].mod;
				FSOUND_STREAM *stream = song[songid].stream;

				if (mod)
					FMUSIC_SetMasterVolume(mod, SendMessage((HWND)lParam, TBM_GETPOS, 0, 0));
				else if (stream)
					FSOUND_SetVolume(song[songid].channel, SendMessage((HWND)lParam, TBM_GETPOS, 0, 0));
			}
			break;
		}

		case WM_VSCROLL:	// echo slider
		{
			if ((HWND)lParam == GetDlgItem(hwnd,IDC_CDVOLUME))
			{
				FSOUND_CD_SetVolume(255 - SendMessage((HWND)lParam, TBM_GETPOS, 0, 0));
				break;
			}
			else if ((HWND)lParam == GetDlgItem(hwnd,IDC_ECHOSLIDER))
			{
				DSP_EchoLen = SendMessage((HWND)lParam, TBM_GETPOS, 0, 0) * outputfreq / 1000;
				if (DSP_EchoLen < FSOUND_DSP_GetBufferLength())
					DSP_EchoLen = FSOUND_DSP_GetBufferLength();

				DSP_EchoOffset = 0;
				memset(DSP_EchoBuffer, 0, MAXECHOLEN);	// echolen is in samples.
				break;
			}
			else if ((HWND)lParam == GetDlgItem(hwnd,IDC_CUTOFFSLIDER))
			{
				DSP_LowPassCutoffFrequency = 520 - (float)SendMessage((HWND)lParam, TBM_GETPOS, 0, 0);
				DSP_LowPassCutoffFrequency *= 10;	// 200 to 5000
				LowPass_Update(DSP_LowPassResonance, DSP_LowPassCutoffFrequency, FSOUND_GetOutputRate());
				break;
			}
			else if ((HWND)lParam == GetDlgItem(hwnd,IDC_RESOSLIDER))
			{
				DSP_LowPassResonance = 520 - (float)SendMessage((HWND)lParam, TBM_GETPOS, 0, 0);
				DSP_LowPassResonance /= 50;			// .2 to 10
				LowPass_Update(DSP_LowPassResonance, DSP_LowPassCutoffFrequency, FSOUND_GetOutputRate());
		
				break;
			}
		}

		case WM_COMMAND:
		{
			HWND lbhwnd = GetDlgItem(hwnd,IDC_SONGLIST);

			switch (LOWORD(wParam))
			{
				case IDCANCEL:
				{
					CloseDown();
					EndDialog(hwnd,0);
		            PostQuitMessage(0);
					return TRUE;
				}
				case IDC_LOAD:		// LOAD MOD
				{
					LoadSong(hwnd, lbhwnd);
					break;
				}
				case IDC_DELETE:		// DELETE MOD
				{
					int songid, count;

					songid = SendMessage(lbhwnd, LB_GETCURSEL, 0, 0);
					
					if (songid == LB_ERR) 
						MessageBox(hwnd, "Error.  Please SELECT a song to delete", "Deleting a mod", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
					else
					{
						// delete the mod from the list
						if (song[songid].mod)
						{
							FMUSIC_FreeSong(song[songid].mod);
							song[songid].mod = NULL;
						}
						else if (song[songid].stream)
						{
							FSOUND_Stream_Close(song[songid].stream);
							song[songid].stream = NULL;
						}

						song[songid].channel = -1;

						SendMessage(lbhwnd, LB_DELETESTRING, songid, 0);

						// shuffle down tunes
						for (count=songid+1; count<MAXSONGS; count++) 
							song[count-1] = song[count];

						if (songid >= SendMessage(lbhwnd, LB_GETCOUNT, 0, 0))
						{
							songid = SendMessage(lbhwnd, LB_GETCOUNT, 0, 0) - 1;
							if (songid < 0) 
								songid = 0;
						}

						SendMessage(lbhwnd, LB_SETCURSEL, songid, 0);
	
						playlistsong = 0;
					}
					break;
				}
				case IDC_PLAY:		// PLAY MOD
				{
					int songid;

					songid = SendMessage(lbhwnd, LB_GETCURSEL, 0, 0);

					if (songid != LB_ERR) 
					{
						if (Button_GetCheck(GetDlgItem(hwnd,IDC_PLAYLIST))) 
							playlistsong = songid;

						if (song[songid].mod)
						{
							if (!FMUSIC_PlaySong(song[songid].mod)) 
								MessageBox(hwnd, "Error.  Cannot start song", "Playing a song", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
						}
						else if (song[songid].stream)
						{
							int channel = FSOUND_Stream_Play(FSOUND_FREE, song[songid].stream);
							if (channel < 0)
								MessageBox(hwnd, "Error.  Cannot start song", "Playing a song", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
							else
							{
								song[songid].channel = channel;
								FSOUND_SetPan(song[songid].channel, FSOUND_STEREOPAN);
								FSOUND_SetVolume(song[songid].channel, 255);
							}

						}

					}
					else 
						MessageBox(hwnd, "Error.  Please select a song to play first", "Playing a song", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);

					break;
				}
				case IDC_STOP:		// STOP MOD
				{
					int songid;

					songid = SendMessage(lbhwnd, LB_GETCURSEL, 0, 0);

					if (songid != LB_ERR) 
					{
						if (song[songid].mod)
							FMUSIC_StopSong(song[songid].mod);
						else if (song[songid].stream)
						{
							FSOUND_Stream_Stop(song[songid].stream);
						}
						song[songid].channel = -1;
					}
					else 
						MessageBox(hwnd, "Error.  Please select a song to play first", "Playing a mod", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);

					break;
				}
				case IDC_EXIT:		// EXIT
				{
			        PostMessage(hwnd, WM_CLOSE, 0, 0);
			        break;
				}
				case IDC_ABOUT:		// ABOUT
				{
					char s[1024];
					char s1[256];
					char s2[256];

					strcpy(s1, "FSOUND Output Method : ");
					switch (FSOUND_GetOutput())
					{
						case FSOUND_OUTPUT_NOSOUND:	strcat(s1, "FSOUND_OUTPUT_NOSOUND\n"); break;
						case FSOUND_OUTPUT_WINMM:	strcat(s1, "FSOUND_OUTPUT_WINMM\n"); break;
						case FSOUND_OUTPUT_DSOUND:	strcat(s1, "FSOUND_OUTPUT_DSOUND\n"); break;
						case FSOUND_OUTPUT_A3D:		strcat(s1, "FSOUND_OUTPUT_A3D\n"); break;
					};

					strcpy(s2, "FSOUND Mixer         : ");
					switch (FSOUND_GetMixer())
					{
						case FSOUND_MIXER_BLENDMODE:	strcat(s2, "FSOUND_MIXER_BLENDMODE\n"); break;
						case FSOUND_MIXER_MMXP5:		strcat(s2, "FSOUND_MIXER_MMXP5\n"); break;
						case FSOUND_MIXER_MMXP6:		strcat(s2, "FSOUND_MIXER_MMXP6\n"); break;
						case FSOUND_MIXER_QUALITY_FPU:	strcat(s2, "FSOUND_MIXER_QUALITY_FPU\n"); break;
						case FSOUND_MIXER_QUALITY_MMXP5:strcat(s2, "FSOUND_MIXER_QUALITY_MMXP5\n"); break;
						case FSOUND_MIXER_QUALITY_MMXP6:strcat(s2, "FSOUND_MIXER_QUALITY_MMXP6\n"); break;
					};
					printf("FSOUND Driver        : ");
					printf("%s\n", FSOUND_GetDriverName(FSOUND_GetDriver()));
					printf("Hardware 3D channels : %d\n", FSOUND_GetNumHardwareChannels());

					sprintf(s, "Copyright (c) 1994-2000, Firelight Multimedia\n\n%s\n%s\nFASTCD Player - A *non polling* player, stops cd hardware interrupting foreground applications.\n\nContact Details:\nemail: support@fmod.org\nurl: http://www.fmod.org", s1, s2);

			        MessageBox(hwnd, s, "FMOD Music System", MB_ICONQUESTION|MB_OK|MB_SYSTEMMODAL);
			        break;
				}
				case IDC_ORDER_DEC:
				{
					int songid = SendMessage(lbhwnd, LB_GETCURSEL, 0, 0);
					if (songid != LB_ERR)
					{
						FMUSIC_MODULE *mod = song[songid].mod;
						FSOUND_STREAM *stream = song[songid].stream;

						if (mod)
							FMUSIC_SetOrder(mod, FMUSIC_GetOrder(mod)-1);
					}
				}	break;
				case IDC_ORDER_INC:
				{
					int songid = SendMessage(lbhwnd, LB_GETCURSEL, 0, 0);
					
					if (songid != LB_ERR)
					{
						FMUSIC_MODULE *mod = song[songid].mod;
						FSOUND_STREAM *stream = song[songid].stream;

						if (mod)
							FMUSIC_SetOrder(mod, FMUSIC_GetOrder(mod)+1);
					}
				}	break;
				case IDC_CONFIG:		// CONFIG
				{
					CloseDSP();

					// remember .. FSOUND_Close cleans up all DSP units, so if you still have pointers to them, 
					// it could cause problems (crashes) because they are pointing to freed data.
					// This is why there is a call to CloseDSP above.
					FSOUND_Close();	

					// call up dialog box to select sound options
					SoundDriver_Init(&outputfreq);

					FSOUND_SetHWND(hwnd);
					
					FSOUND_SetBufferSize(200);

					// initialize FSOUND
					if (!FSOUND_Init(outputfreq,NUMCHANNELS, 0))
					{
    					MessageBox(hwnd, FMOD_ErrorString(FSOUND_GetError()), "FSOUND", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
    					return FALSE;
					}

					InitDSP();

					break;
				}

				//= CD COMMANDS ============================================================
				case IDC_CDPLAY:		
				{
					// if play is pressed and it is paused, then just unpause.
					if (FSOUND_CD_GetPaused()) 
					{
						FSOUND_CD_SetPaused(FALSE);
						break;
					}

					if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIORANDOM)))
					{
						if (FSOUND_CD_GetNumTracks())
							FSOUND_CD_Play((rand()%FSOUND_CD_GetNumTracks())+1);
					}
					else
						FSOUND_CD_Play(FSOUND_CD_GetTrack() == 0 ? 1 : FSOUND_CD_GetTrack());
			        break;
				}
				case IDC_CDSTOP:
				{
			        FSOUND_CD_Stop();
			        break;
				}
				case IDC_CDPAUSE:
				{
			        FSOUND_CD_SetPaused((char)!FSOUND_CD_GetPaused());
			        break;
				}
				case IDC_CDBACK:
				{
			        FSOUND_CD_Play(FSOUND_CD_GetTrack()-1);
			        break;
				}
				case IDC_CDFORWARD:
				{
			        FSOUND_CD_Play(FSOUND_CD_GetTrack()+1);
			        break;
				}
				case IDC_CDEJECT:
				{
					FSOUND_CD_Eject();
					break;
				}
				case IDC_CDMINIMIZE:
				{
					Button_SetCheck(GetDlgItem(cdhwnd,IDC_RADIOCONTINUOUS), FALSE);
					Button_SetCheck(GetDlgItem(cdhwnd,IDC_RADIOLOOPED), FALSE);
					Button_SetCheck(GetDlgItem(cdhwnd,IDC_RADIORANDOM), FALSE);

					// Check CD LOOPED checkbox
					if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIOCONTINUOUS)))
						Button_SetCheck(GetDlgItem(cdhwnd,IDC_RADIOCONTINUOUS), TRUE);
					else if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIOLOOPED)))
						Button_SetCheck(GetDlgItem(cdhwnd,IDC_RADIOLOOPED), TRUE);
					else if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIORANDOM)))
						Button_SetCheck(GetDlgItem(cdhwnd,IDC_RADIORANDOM), TRUE);

					// set position of the cd volume slider
					SendMessage(GetDlgItem(cdhwnd,IDC_CDVOLUME), TBM_SETPOS, TRUE, 255-FSOUND_CD_GetVolume());

					ShowWindow(mainhwnd, SW_HIDE);
					ShowWindow(cdhwnd, SW_SHOW);

					setting_cdmode = TRUE;
					break;
				}
			}
			break;
		}
		case WM_DROPFILES :
		{
			HDROP			hDrop = (HDROP)wParam;
			int				numfiles, count, currsong;
			char			name[1024];
			FMUSIC_MODULE	*mod;
			FSOUND_STREAM	*stream;
			HWND			LB_Hwnd = GetDlgItem(hwnd,IDC_SONGLIST);

			numfiles = DragQueryFile(hDrop, 0xFFFFFFFF, &name[0], 1024);
			if (!numfiles) 
				break;

			for (count=0; count<numfiles; count++)
			{
				DragQueryFile(hDrop, count, &name[0], 1024);
			
				// Open the file.
				stream = NULL;
				mod = FMUSIC_LoadSong(name);
				if (!mod)
					stream = FSOUND_Stream_OpenFile(name, FSOUND_NORMAL | FSOUND_LOOP_NORMAL | FSOUND_2D, 0);

				if (!mod && !stream)
				{
					MessageBox(hwnd, FMOD_ErrorString(FSOUND_GetError()), "Loading a mod", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
					break;
				}
				currsong = SendMessage(LB_Hwnd, LB_GETCOUNT, 0, 0);
				if (currsong >= MAXSONGS) 
				{
					MessageBox(hwnd, "Error.  Unable to fit any more mods in.. please restart app", "Loading a mod", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
					break;
				}

				song[currsong].mod = mod;
				song[currsong].stream = stream;

				SendMessage(LB_Hwnd, LB_ADDSTRING, 0, (LPARAM) (LPCTSTR)name);
				UpdateWindow(LB_Hwnd);
				SendMessage(LB_Hwnd, LB_SETCURSEL, currsong, 0);
			}

			DragFinish(hDrop);
			break;
		}
		case WM_TIMER:
		{
			char			s[256];
			HDC				hdc = GetDC(hwnd);
			HFONT			myfont;
			FMUSIC_MODULE	*mod;
			FSOUND_STREAM	*stream;
			int				songid;
			int				greyoutfilters;

			greyoutfilters = FALSE;

			myfont = GetStockObject(DEFAULT_GUI_FONT);
			SelectObject(hdc, myfont);

			SetBkColor(hdc, GetSysColor(COLOR_3DFACE));
			SetTextColor(hdc, RGB(0, 0, 255));

			sprintf(s, "%03d", FSOUND_GetChannelsPlaying());
			TextOut(hdc, (int)(636*scalex), (int)(202*scaley), s, strlen(s));

			sprintf(s, "%5.02f%% ", FSOUND_GetCPUUsage());
			TextOut(hdc, (int)(626*scalex), (int)(234*scaley), s, strlen(s));

			sprintf(s, "%02d / %02d ", FSOUND_CD_GetTrack(), FSOUND_CD_GetNumTracks());
			TextOut(hdc, (int)(284*scalex), (int)(222*scaley), s, strlen(s));
			sprintf(s, FSOUND_CD_GetPaused() ? "paused" : "ready    ");
			TextOut(hdc, (int)(284*scalex), (int)(235*scaley), s, strlen(s));
				
			//= DSP BUTTONS ================================================================
			if (DSP_Ready)
			{
				int count;
			
				// if the unit is inactive and the checkbox is checked, then clear preverb buffer
				// this stops any old preverb dregs hanging around.
				if (!FSOUND_DSP_GetActive(DSP_PreverbTap[0].Unit) && (char)Button_GetCheck(GetDlgItem(hwnd,IDC_PREVERB)))
				{
					for (count=0; count < PREVERB_NUMTAPS; count++)
						memset(DSP_PreverbTap[count].historybuff, 0, DSP_PreverbTap[count].historylen<<2);	// preverblen is in samples.
				}
				for (count=0; count < PREVERB_NUMTAPS; count++)
					FSOUND_DSP_SetActive(DSP_PreverbTap[count].Unit, (char)Button_GetCheck(GetDlgItem(hwnd,IDC_PREVERB)));

				// if the unit is inactive and the checkbox is checked, then clear reverb buffer
				// this stops any old reverb dregs hanging around.
				if (!FSOUND_DSP_GetActive(DSP_ReverbTap[0].Unit) && (char)Button_GetCheck(GetDlgItem(hwnd,IDC_REVERB)))
				{
					for (count=0; count < REVERB_NUMTAPS; count++)
						memset(DSP_ReverbTap[count].historybuff, 0, DSP_ReverbTap[count].historylen<<2);	// preverblen is in samples.
				}
				for (count=0; count < REVERB_NUMTAPS; count++)
				{
					FSOUND_DSP_SetActive(DSP_ReverbTap[count].Unit, (char)Button_GetCheck(GetDlgItem(hwnd,IDC_REVERB)));
				}

				
				// MIDI reverb uses the reverb button as well
				FMUSIC_SetReverb((char)Button_GetCheck(GetDlgItem(hwnd,IDC_REVERB)));


				// if the unit is inactive and the checkbox is checked, then clear echo buffer
				// this stops any old echo dregs hanging around.
				if (!FSOUND_DSP_GetActive(DSP_EchoUnit) && (char)Button_GetCheck(GetDlgItem(hwnd,IDC_ECHO)))
				{
					// set status
					ShowWindow(GetDlgItem(mainhwnd,IDC_ECHOSLIDER), SW_SHOW);
					memset(DSP_EchoBuffer, 0, MAXECHOLEN);	// echolen is in samples.

					SendMessage(GetDlgItem(mainhwnd,IDC_ECHOSLIDER), TBM_SETPOS, TRUE, DSP_EchoLen * 1000 / outputfreq);

				}
				if (FSOUND_DSP_GetActive(DSP_EchoUnit) && !(char)Button_GetCheck(GetDlgItem(hwnd,IDC_ECHO)))
					ShowWindow(GetDlgItem(mainhwnd,IDC_ECHOSLIDER), SW_HIDE);

				FSOUND_DSP_SetActive(DSP_EchoUnit, (char)Button_GetCheck(GetDlgItem(hwnd,IDC_ECHO)));



				if (!FSOUND_DSP_GetActive(DSP_LowPassUnit) && (char)Button_GetCheck(GetDlgItem(hwnd,IDC_LOWPASS)))
				{
					// set status
					ShowWindow(GetDlgItem(mainhwnd,IDC_CUTOFFSLIDER), SW_SHOW);
					ShowWindow(GetDlgItem(mainhwnd,IDC_RESOSLIDER), SW_SHOW);

					SendMessage(GetDlgItem(mainhwnd,IDC_CUTOFFSLIDER), TBM_SETPOS, TRUE, 520 - (int)(DSP_LowPassCutoffFrequency / 10.0f));
					SendMessage(GetDlgItem(mainhwnd,IDC_RESOSLIDER), TBM_SETPOS, TRUE, 520 - (int)(DSP_LowPassResonance * 50.0f));
				}
				if (FSOUND_DSP_GetActive(DSP_LowPassUnit) && !(char)Button_GetCheck(GetDlgItem(hwnd,IDC_LOWPASS)))
				{
					ShowWindow(GetDlgItem(mainhwnd,IDC_CUTOFFSLIDER), SW_HIDE);
					ShowWindow(GetDlgItem(mainhwnd,IDC_RESOSLIDER), SW_HIDE);
				}

				FSOUND_DSP_SetActive(DSP_LowPassUnit, (char)Button_GetCheck(GetDlgItem(hwnd,IDC_LOWPASS)));
			}

			// Check CD LOOPED checkbox
			if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIOCONTINUOUS)))
				FSOUND_CD_SetPlayMode(FSOUND_CD_PLAYCONTINUOUS);
			else if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIOLOOPED)))
				FSOUND_CD_SetPlayMode(FSOUND_CD_PLAYLOOPED);
			else if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIORANDOM)))
				FSOUND_CD_SetPlayMode(FSOUND_CD_PLAYRANDOM);

			// Check PLAYLIST checkbox
			if (Button_GetCheck(GetDlgItem(hwnd,IDC_PLAYLIST)))
			{
				mod = song[playlistsong].mod;
				if (mod)
				{
					if (FMUSIC_IsFinished(mod))
					{
						FMUSIC_StopSong(mod);
						playlistsong++;

						if (playlistsong>= SendMessage(GetDlgItem(hwnd,IDC_SONGLIST), LB_GETCOUNT, 0, 0)) 
							playlistsong = 0;

						if (song[playlistsong].mod)
							FMUSIC_PlaySong(song[playlistsong].mod);
						if (song[playlistsong].stream)
							song[playlistsong].channel = FSOUND_Stream_Play(FSOUND_FREE, song[playlistsong].stream);

						SendMessage(GetDlgItem(hwnd,IDC_SONGLIST), LB_SETCURSEL, playlistsong, 0);
					}
				}
				else 
				{
					stream = song[playlistsong].stream;
					if (stream)
					{
						int off = FSOUND_Stream_GetTime(stream);
						int len = FSOUND_Stream_GetLengthMs(stream);

						if (off >= len)
						{
							FSOUND_Stream_Stop(stream);
							off = FSOUND_Stream_GetTime(stream);
							playlistsong++;

							if (playlistsong>= SendMessage(GetDlgItem(hwnd,IDC_SONGLIST), LB_GETCOUNT, 0, 0)) 
								playlistsong = 0;

							if (song[playlistsong].mod)
								FMUSIC_PlaySong(song[playlistsong].mod);
							if (song[playlistsong].stream)
								song[playlistsong].channel = FSOUND_Stream_Play(FSOUND_FREE, song[playlistsong].stream);

							SendMessage(GetDlgItem(hwnd,IDC_SONGLIST), LB_SETCURSEL, playlistsong, 0);
						}
					}
				}
			}

			songid = SendMessage(GetDlgItem(hwnd,IDC_SONGLIST), LB_GETCURSEL, 0, 0);

			if (songid >= 0)
			{
				mod = song[songid].mod;
				if (mod)
				{
					char *type[] = 
					{
						"Unknown                      ",
						"Protracker / FastTracker     ",
						"ScreamTracker 3              ",
						"FastTracker 2                ",
						"Impulse Tracker              ",
						"MIDI                         ",
					};

					strcpy(s, FMUSIC_GetName(mod));

					TextOut(hdc, (int)(477*scalex), (int)(27*scaley), s, strlen(s));
					TextOut(hdc, (int)(477*scalex), (int)(44*scaley), type[FMUSIC_GetType(mod)], strlen(type[FMUSIC_GetType(mod)]));
					sprintf(s, "%02d", FMUSIC_GetSpeed(mod));
					TextOut(hdc, (int)(477*scalex), (int)(61*scaley), s, strlen(s));
					sprintf(s, "%03d", FMUSIC_GetBPM(mod));
					TextOut(hdc, (int)(477*scalex), (int)(78*scaley), s, strlen(s));
					sprintf(s, "%03d / %03d", FMUSIC_GetOrder(mod), FMUSIC_GetNumOrders(mod));
					TextOut(hdc, (int)(477*scalex), (int)(95*scaley), s, strlen(s));
					sprintf(s, "%03d / %03d", FMUSIC_GetPattern(mod), FMUSIC_GetNumPatterns(mod));
					TextOut(hdc, (int)(477*scalex), (int)(112*scaley), s, strlen(s));
					sprintf(s, "%03d / %03d", FMUSIC_GetRow(mod), FMUSIC_GetPatternLength(mod, FMUSIC_GetOrder(mod)));
					TextOut(hdc, (int)(477*scalex), (int)(129*scaley), s, strlen(s));

					SendMessage(GetDlgItem(hwnd,IDC_PROGRESS1), PBM_SETRANGE, 0, MAKELPARAM(0, FMUSIC_GetNumOrders(mod)-1));
					SendMessage(GetDlgItem(hwnd,IDC_PROGRESS1), PBM_SETPOS, FMUSIC_GetOrder(mod), 0);

					// grey out fx buttons for midi
					if (FMUSIC_GetType(mod) == FMUSIC_TYPE_MIDI)
						greyoutfilters = TRUE;
				}
				else
				{
					SendMessage(GetDlgItem(hwnd,IDC_PROGRESS1), PBM_SETRANGE, 0, MAKELPARAM(0, 100) );
					SendMessage(GetDlgItem(hwnd,IDC_PROGRESS1), PBM_SETPOS, (WPARAM)((float)FSOUND_Stream_GetPosition(song[songid].stream) / (float)FSOUND_Stream_GetLength(song[songid].stream) * 100.0f), 0);
				}

				// set position of the master volume slider
				if (mod)
					SendMessage(GetDlgItem(hwnd,IDC_SLIDER1), TBM_SETPOS, TRUE, FMUSIC_GetMasterVolume(mod));
				else
					SendMessage(GetDlgItem(hwnd,IDC_SLIDER1), TBM_SETPOS, TRUE, FSOUND_GetVolume(song[songid].channel));
			}

			if (greyoutfilters)
			{
				EnableWindow(GetDlgItem(hwnd,IDC_LOWPASS), FALSE);
				EnableWindow(GetDlgItem(hwnd,IDC_PREVERB), FALSE);
				EnableWindow(GetDlgItem(hwnd,IDC_ECHO), FALSE);
			}
			else
			{
				EnableWindow(GetDlgItem(hwnd,IDC_LOWPASS), TRUE);
				EnableWindow(GetDlgItem(hwnd,IDC_PREVERB), TRUE);
				EnableWindow(GetDlgItem(hwnd,IDC_ECHO), TRUE);
			}

			DeleteObject(myfont);
			ReleaseDC(hwnd, hdc);
			break;
		}
		default:
			return FALSE;
	}

	return FALSE;
}



/*
[
	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
]
*/
BOOL CALLBACK FMOD_CDDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg)
	{
		case WM_INITDIALOG:
		{
		
			//===================================================================================
			// SET UP A PAINT TIMER FOR INTERFACE
			//===================================================================================
			cdtimerid = SetTimer(hwnd, 0, 100, 0L); 
			if (!cdtimerid)
			{
    			MessageBox( NULL, "Too many timers in use", "Error", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
    			return FALSE;
			}

			SetFocus(hwnd);
		
			return TRUE;
		}

		case WM_CLOSE:
		{
			KillTimer(hwnd, cdtimerid);
			EndDialog(hwnd,0);
			SendMessage(mainhwnd, WM_CLOSE, 0, 0);
			return TRUE;
		}


		case WM_VSCROLL:
		{
			if ((HWND)lParam == GetDlgItem(hwnd,IDC_CDVOLUME))
			{
				FSOUND_CD_SetVolume(255 - SendMessage((HWND)lParam, TBM_GETPOS, 0, 0));
				break;
			}
		}

		case WM_COMMAND:
		{
			switch (LOWORD(wParam))
			{
				case IDCANCEL:
				{
					EndDialog(hwnd,0);
					return TRUE;
				}

				//= CD COMMANDS ============================================================
				case IDC_CDPLAY:		
				{
					// if play is pressed and it is paused, then just unpause.
					if (FSOUND_CD_GetPaused()) 
					{
						FSOUND_CD_SetPaused(FALSE);
						break;
					}

					if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIORANDOM)))
					{
						if (FSOUND_CD_GetNumTracks())
							FSOUND_CD_Play((rand()%FSOUND_CD_GetNumTracks())+1);
					}
					else
						FSOUND_CD_Play(FSOUND_CD_GetTrack() == 0 ? 1 : FSOUND_CD_GetTrack());
			        break;
				}
				case IDC_CDSTOP:
				{
			        FSOUND_CD_Stop();
			        break;
				}
				case IDC_CDPAUSE:
				{
			        FSOUND_CD_SetPaused((char)!FSOUND_CD_GetPaused());
			        break;
				}
				case IDC_CDBACK:
				{
			        FSOUND_CD_Play(FSOUND_CD_GetTrack()-1);
			        break;
				}
				case IDC_CDFORWARD:
				{
			        FSOUND_CD_Play(FSOUND_CD_GetTrack()+1);
			        break;
				}
				case IDC_CDEJECT:
				{
					FSOUND_CD_Eject();
					break;
				}
				case IDC_CDMAXIMIZE:
				{
					Button_SetCheck(GetDlgItem(mainhwnd,IDC_RADIOCONTINUOUS), FALSE);
					Button_SetCheck(GetDlgItem(mainhwnd,IDC_RADIOLOOPED), FALSE);
					Button_SetCheck(GetDlgItem(mainhwnd,IDC_RADIORANDOM), FALSE);

					// Check CD LOOPED checkbox
					if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIOCONTINUOUS)))
						Button_SetCheck(GetDlgItem(mainhwnd,IDC_RADIOCONTINUOUS), TRUE);
					else if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIOLOOPED)))
						Button_SetCheck(GetDlgItem(mainhwnd,IDC_RADIOLOOPED), TRUE);
					else if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIORANDOM)))
						Button_SetCheck(GetDlgItem(mainhwnd,IDC_RADIORANDOM), TRUE);

					// set position of the cd volume slider
					SendMessage(GetDlgItem(mainhwnd,IDC_CDVOLUME), TBM_SETPOS, TRUE, 255-FSOUND_CD_GetVolume());

					ShowWindow(mainhwnd, SW_SHOW);
					ShowWindow(cdhwnd, SW_HIDE);
					setting_cdmode = FALSE;

					break;
				}
			}
			break;
		}
		case WM_DROPFILES :
		{
			HDROP			hDrop = (HDROP)wParam;
			int				numfiles, count, currsong;
			char			name[1024];
			FMUSIC_MODULE	*mod;
			FSOUND_STREAM	*stream;
			HWND			LB_Hwnd = GetDlgItem(hwnd,IDC_SONGLIST);

			numfiles = DragQueryFile(hDrop, 0xFFFFFFFF, &name[0], 1024);
			if (!numfiles) break;

			for (count=0; count<numfiles; count++)
			{
				DragQueryFile(hDrop, count, &name[0], 1024);
			
				stream = NULL;
				mod = FMUSIC_LoadSong(name);
				if (!mod)
					stream = FSOUND_Stream_OpenFile(name, FSOUND_NORMAL | FSOUND_LOOP_NORMAL | FSOUND_2D, 0);

				if (!mod && !stream)
				{
					MessageBox(hwnd, FMOD_ErrorString(FSOUND_GetError()), "Loading a mod", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
					break;
				}
				currsong = SendMessage(LB_Hwnd, LB_GETCOUNT, 0, 0);
				if (currsong >= MAXSONGS) 
				{
					MessageBox(hwnd, "Error.  Unable to fit any more mods in.. please restart app", "Loading a mod", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
					break;
				}
				song[currsong].mod = mod;

				
				SendMessage(LB_Hwnd, LB_ADDSTRING, 0, (LPARAM) (LPCTSTR)name);
				UpdateWindow(LB_Hwnd);
				SendMessage(LB_Hwnd, LB_SETCURSEL, currsong, 0);
			}

			DragFinish(hDrop);
			break;
		}
		case WM_TIMER:
		{
			char	s[256];
			HDC		hdc = GetDC(hwnd);
			HFONT	myfont;

			myfont = GetStockObject(DEFAULT_GUI_FONT);
			SelectObject(hdc, myfont);

			SetBkColor(hdc, GetSysColor(COLOR_3DFACE));
			SetTextColor(hdc, RGB(0, 0, 255));

			sprintf(s, "%02d / %02d ", FSOUND_CD_GetTrack(), FSOUND_CD_GetNumTracks());
			TextOut(hdc, (int)(191*scalex), (int)(31*scaley), s, strlen(s));
			sprintf(s, FSOUND_CD_GetPaused() ? "paused" : "ready    ");
			TextOut(hdc, (int)(191*scalex), (int)(44*scaley), s, strlen(s));

			// Check CD LOOPED checkbox
			if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIOCONTINUOUS)))
				FSOUND_CD_SetPlayMode(FSOUND_CD_PLAYCONTINUOUS);
			else if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIOLOOPED)))
				FSOUND_CD_SetPlayMode(FSOUND_CD_PLAYLOOPED);
			else if (Button_GetCheck(GetDlgItem(hwnd,IDC_RADIORANDOM)))
				FSOUND_CD_SetPlayMode(FSOUND_CD_PLAYRANDOM);
				

			DeleteObject(myfont);
			ReleaseDC(hwnd, hdc);
			break;
		}
		default:
			return FALSE;
	}

	return FALSE;
}




/*
[API]
[
	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
]
*/
char SetupInterface(HINSTANCE hinst, LPSTR lpCmdLine)
{
	FMUSIC_MODULE	*mod;
	FSOUND_STREAM	*stream;
	char			name[256];
	RECT			r;
	int				desktop_height,desktop_width;
	int				window_height,window_width;

	LoadSettings();

	GetWindowRect(GetDesktopWindow(), &r);
	desktop_width = r.right - r.left;
	desktop_height = r.bottom - r.top;
	
	// fix up screwed up xy positions
	if (setting_xpos > desktop_width || setting_xpos < 0)
		setting_xpos = (desktop_width / 2)-320;
	if (setting_ypos > desktop_height || setting_ypos < 0)
		setting_ypos = (desktop_height / 2)-140;

	if (setting_cdxpos > desktop_width || setting_cdxpos < 0)
		setting_cdxpos = (desktop_width / 2)-320;
	if (setting_cdypos > desktop_height || setting_cdypos < 0)
		setting_cdypos = (desktop_height / 2)-140;

	mainhwnd = CreateDialog(hinst,MAKEINTRESOURCE(IDD_INTERFACE),GetDesktopWindow(), FMOD_DlgProc);

	cdhwnd = CreateDialog(hinst,MAKEINTRESOURCE(IDD_CDINTERFACE),GetDesktopWindow(), FMOD_CDDlgProc);

	ShowWindow(mainhwnd, SW_HIDE);
	ShowWindow(cdhwnd, SW_HIDE);

	// set the icon
	SetClassLong(mainhwnd, GCL_HICON, (LONG) LoadIcon(hinst, MAKEINTRESOURCE(IDI_ICON1)));


	GetWindowRect(mainhwnd, &r);
	MoveWindow(mainhwnd, setting_xpos, setting_ypos, r.right, r.bottom, TRUE);

	window_width = r.right - r.left;
	window_height = r.bottom - r.top;
	scalex = (float)window_width / 675.0f;
	scaley = (float)window_height / 289.0f;

	GetWindowRect(cdhwnd, &r);
	MoveWindow(cdhwnd, setting_cdxpos, setting_cdypos, r.right, r.bottom, TRUE);
//	MoveWindow(cdhwnd, setting_xpos+93, setting_ypos+191, r.right, r.bottom, TRUE);

	
	if (!setting_cdmode) 
		ShowWindow(mainhwnd, SW_SHOW);
	else 
		ShowWindow(cdhwnd, SW_SHOW);

	InitCommonControls();

	// set the range of the master volume slider
	SendMessage(GetDlgItem(mainhwnd,IDC_SLIDER1), TBM_SETRANGE, TRUE, MAKELPARAM(0, 256));

	// set position of the master volume slider
	SendMessage(GetDlgItem(mainhwnd,IDC_SLIDER1), TBM_SETPOS, TRUE, 256);

	// set the range of the echo slider
	// set position of the echo slider
	// set status
	SendMessage(GetDlgItem(mainhwnd,IDC_ECHOSLIDER), TBM_SETRANGE, TRUE, MAKELPARAM(20, 500));
	SendMessage(GetDlgItem(mainhwnd,IDC_ECHOSLIDER), TBM_SETPOS, TRUE, 500);
	ShowWindow(GetDlgItem(mainhwnd,IDC_ECHOSLIDER), SW_HIDE);

	// set the range of the cutoff freq slider
	// set position of the cutoff freq  slider
	// set status
	SendMessage(GetDlgItem(mainhwnd,IDC_CUTOFFSLIDER), TBM_SETRANGE, TRUE, MAKELPARAM(20, 500));
	SendMessage(GetDlgItem(mainhwnd,IDC_CUTOFFSLIDER), TBM_SETPOS, TRUE, 20);
	ShowWindow(GetDlgItem(mainhwnd,IDC_CUTOFFSLIDER), SW_HIDE);

	// set the range of the resonance slider
	// set position of the resonance slider
	// set status
	SendMessage(GetDlgItem(mainhwnd,IDC_RESOSLIDER), TBM_SETRANGE, TRUE, MAKELPARAM(20, 500));
	SendMessage(GetDlgItem(mainhwnd,IDC_RESOSLIDER), TBM_SETPOS, TRUE, 500);
	ShowWindow(GetDlgItem(mainhwnd,IDC_RESOSLIDER), SW_HIDE);

	// set the range of the cd volume slider
	SendMessage(GetDlgItem(mainhwnd,IDC_CDVOLUME), TBM_SETRANGE, TRUE, MAKELPARAM(0, 255));

	// set position of the cd volume slider
	SendMessage(GetDlgItem(mainhwnd,IDC_CDVOLUME), TBM_SETPOS, TRUE, 255- FSOUND_CD_GetVolume());


	//= COMMAND LINE ====================================================================
	if (strlen(lpCmdLine)) 
	{
#if 0
		MessageBox(mainhwnd, lpCmdLine, "Command line parameter", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
#endif

		strcpy(name, lpCmdLine);

		// Open the file.
		stream = NULL;
		mod = FMUSIC_LoadSong(name);
		if (!mod)
			stream = FSOUND_Stream_OpenFile(name, FSOUND_NORMAL | FSOUND_LOOP_NORMAL | FSOUND_2D, 0);

		if (!mod && !stream) 
			MessageBox(mainhwnd, FMOD_ErrorString(FSOUND_GetError()), "Loading a song", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
		else
		{
			HWND LB_Hwnd = GetDlgItem(mainhwnd,IDC_SONGLIST);
			int currsong;

			currsong = SendMessage(LB_Hwnd, LB_GETCOUNT, 0, 0);
			if (currsong >= MAXSONGS) 
				MessageBox(mainhwnd, "Error.  Unable to fit any more mods in.. please restart app", "Loading a mod", MB_ICONHAND|MB_OK|MB_SYSTEMMODAL);
			else
			{
				song[currsong].mod = mod;
				song[currsong].stream = stream;

				SendMessage(LB_Hwnd, LB_ADDSTRING, 0, (LPARAM) (LPCTSTR)name);
				UpdateWindow(LB_Hwnd);
				SendMessage(LB_Hwnd, LB_SETCURSEL, currsong, 0);
			}
		}
	}

	srand(clock());
	Button_SetCheck(GetDlgItem(mainhwnd,IDC_RADIOCONTINUOUS), TRUE);
	Button_SetCheck(GetDlgItem(cdhwnd,IDC_RADIOCONTINUOUS), TRUE);

	// set the range of the cd volume slider
	SendMessage(GetDlgItem(cdhwnd,IDC_CDVOLUME), TBM_SETRANGE, TRUE, MAKELPARAM(0, 255));

	// set position of the cd volume slider
	SendMessage(GetDlgItem(cdhwnd,IDC_CDVOLUME), TBM_SETPOS, TRUE, 255 - FSOUND_CD_GetVolume());

	return TRUE;
}

/*
[
	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
]
*/
void CloseDown() 
{
	int count;
	RECT r;
	int desktop_height, desktop_width;

	GetWindowRect(mainhwnd, &r);
	setting_xpos = r.left;
	setting_ypos = r.top;
	GetWindowRect(cdhwnd, &r);
	setting_cdxpos = r.left;
	setting_cdypos = r.top;

	GetWindowRect(GetDesktopWindow(), &r);
	desktop_width = r.right - r.left;
	desktop_height = r.bottom - r.top;
	if (setting_xpos > desktop_width || setting_xpos < 0)
		setting_xpos = (desktop_width / 2)-320;
	if (setting_ypos > desktop_height || setting_ypos < 0)
		setting_ypos = (desktop_height / 2)-140;

	SaveSettings();

	for (count=0; count<MAXSONGS; count++)
		FMUSIC_FreeSong(song[count].mod);
	
	FSOUND_Close();

}


/*
	[NAME]

	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
*/
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow) 
{
    MSG         msg;

	if (FSOUND_GetVersion() < FMOD_VERSION)
		MessageBox(GetForegroundWindow(), "INCORRECT DLL VERSION!!", "FMOD ERROR", MB_OK);

    // Initialize
	if (!SetupInterface(hInstance, lpCmdLine)) return FALSE;

  	while (GetMessage( &msg, 0, 0, 0 ))
	{
    	TranslateMessage( &msg );
      	DispatchMessage( &msg );
  	}

	return( msg.wParam );
}
