/*
	Twilight Prophecy 3D/Multimedia SDK
	A multi-platform development system for virtual reality and multimedia.

	Copyright (C) 1997-2001 by Twilight 3D Finland Oy Ltd.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	Please read the file LICENSE.TXT for additional details.


	source: 
		Matrix4x4 implementation

	revision history:
		Jan/31/2001 - Jukka Liimatta - initial revision
		Jul/13/2001 - Andrew Griffiths - fixed RotateXYZ() and MultRotateXYZ()
*/
#include <prmath/prmath.hpp>
using namespace prmath;



typedef float mat44[4][4];


Matrix4x4::Matrix4x4(float s)
{
	m44[0][0] = s; m44[0][1] = 0; m44[0][2] = 0; m44[0][3] = 0;
	m44[1][0] = 0; m44[1][1] = s; m44[1][2] = 0; m44[1][3] = 0;
	m44[2][0] = 0; m44[2][1] = 0; m44[2][2] = s; m44[2][3] = 0;
	m44[3][0] = 0; m44[3][1] = 0; m44[3][2] = 0; m44[3][3] = 1;
}


Matrix4x4::Matrix4x4(const Matrix4x4& m)
{
	for ( int i=0; i<4; i++ )
	{
		m44[i][0] = m.m44[i][0];
		m44[i][1] = m.m44[i][1];
		m44[i][2] = m.m44[i][2];
		m44[i][3] = m.m44[i][3];
	}
}


Matrix4x4::Matrix4x4(const Quaternion& q)
{
	*this = q;
}


Matrix4x4 Matrix4x4::operator + (const Matrix4x4& right) const
{
	Matrix4x4 r;
	const mat44& m = right.m44;

	for ( int i=0; i<4; i++ )
	{
		r.m44[i][0] = m44[i][0] + m[i][0];
		r.m44[i][1] = m44[i][1] + m[i][1];
		r.m44[i][2] = m44[i][2] + m[i][2];
		r.m44[i][3] = m44[i][3] + m[i][3];
	}
	return r;
}


Matrix4x4 Matrix4x4::operator - (const Matrix4x4& right) const
{
	Matrix4x4 r;
	const mat44& m = right.m44;

	for ( int i=0; i<4; i++ )
	{
		r.m44[i][0] = m44[i][0] - m[i][0];
		r.m44[i][1] = m44[i][1] - m[i][1];
		r.m44[i][2] = m44[i][2] - m[i][2];
		r.m44[i][3] = m44[i][3] - m[i][3];
	}
	return r;
}


Matrix4x4 Matrix4x4::operator * (const Matrix4x4& right) const
{
	Matrix4x4 r;
	const mat44& m = right.m44;
	
	for ( int i=0; i<4; i++ )
	{
		const float* v = m16 + i*4;
		r.m44[i][0] = v[0]*m[0][0] + v[1]*m[1][0] + v[2]*m[2][0] + v[3]*m[3][0];
		r.m44[i][1] = v[0]*m[0][1] + v[1]*m[1][1] + v[2]*m[2][1] + v[3]*m[3][1];
		r.m44[i][2] = v[0]*m[0][2] + v[1]*m[1][2] + v[2]*m[2][2] + v[3]*m[3][2];
		r.m44[i][3] = v[0]*m[0][3] + v[1]*m[1][3] + v[2]*m[2][3] + v[3]*m[3][3];
	}
	return r;
}


Matrix4x4& Matrix4x4::operator += (const Matrix4x4& right)
{
	const mat44& m = right.m44;
	for ( int i=0; i<4; i++ )
	{
		m44[i][0] += m[i][0];
		m44[i][1] += m[i][1];
		m44[i][2] += m[i][2];
		m44[i][3] += m[i][3];
	}
	return *this;
}


Matrix4x4& Matrix4x4::operator -= (const Matrix4x4& right)
{
	const mat44& m = right.m44;
	for ( int i=0; i<4; i++ )
	{
		m44[i][0] -= m[i][0];
		m44[i][1] -= m[i][1];
		m44[i][2] -= m[i][2];
		m44[i][3] -= m[i][3];
	}
	return *this;
}


Matrix4x4& Matrix4x4::operator *= (const Matrix4x4& right)
{
	const mat44& m = right.m44;
	for ( int i=0; i<4; i++ )
	{
		float v[4];
		v[0] = m44[i][0];
		v[1] = m44[i][1];
		v[2] = m44[i][2];
		v[3] = m44[i][3];
		m44[i][0] = v[0]*m[0][0] + v[1]*m[1][0] + v[2]*m[2][0] + v[3]*m[3][0];
		m44[i][1] = v[0]*m[0][1] + v[1]*m[1][1] + v[2]*m[2][1] + v[3]*m[3][1];
		m44[i][2] = v[0]*m[0][2] + v[1]*m[1][2] + v[2]*m[2][2] + v[3]*m[3][2];
		m44[i][3] = v[0]*m[0][3] + v[1]*m[1][3] + v[2]*m[2][3] + v[3]*m[3][3];
	}
	return *this;
}


void Matrix4x4::operator = (const Matrix4x4& right)
{
	const mat44& m = right.m44;
	for ( int i=0; i<4; i++ )
	{
		m44[i][0] = m[i][0];
		m44[i][1] = m[i][1];
		m44[i][2] = m[i][2];
		m44[i][3] = m[i][3];
	}
}


void Matrix4x4::operator = (const Quaternion& q)
{
	float sx = q.x * 2;
	float sy = q.y * 2;
	float sz = q.z * 2;
	float wx = q.w * sx;
	float wy = q.w * sy;
	float wz = q.w * sz;
	float xx = q.x * sx;
	float xy = q.x * sy;
	float xz = q.x * sz;
	float yy = q.y * sy;
	float yz = q.y * sz;
	float zz = q.z * sz;
	
	m44[0][0] = 1 - yy - zz;
	m44[0][1] = xy + wz;
	m44[0][2] = xz - wy;
	m44[0][3] = 0;
	m44[1][0] = xy - wz;
	m44[1][1] = 1 - xx - zz;
	m44[1][2] = yz + wx;
	m44[1][3] = 0;
	m44[2][0] = xz + wy;
	m44[2][1] = yz - wx;
	m44[2][2] = 1 - xx - yy;
	m44[2][3] = 0;
	m44[3][0] = 0;
	m44[3][1] = 0;
	m44[3][2] = 0;
	m44[3][3] = 1;
}


float Matrix4x4::GetDeterminant() const
{
	return
		m44[0][0] * m44[1][1] * m44[2][2] +
		m44[0][1] * m44[1][2] * m44[2][0] +
		m44[0][2] * m44[1][0] * m44[2][1] -
		m44[0][2] * m44[1][1] * m44[2][0] -
		m44[0][1] * m44[1][0] * m44[2][2] -
		m44[0][0] * m44[1][2] * m44[2][1];
}


Vector3 Matrix4x4::GetEuler(EulerOrder euler) const
{
	//--------------------------------------------------------------------------
	// original implementation from the Graphics Gems IV, Academic Press, 1994.
	//--------------------------------------------------------------------------
	#define EulSafe	     "\000\001\002\000"
	#define EulNext	     "\001\002\000\001"
	#define EulGetOrd(ord,i,j,k,n,s,f) {unsigned o=ord;f=o&1;s=o&1; n=o&1;i=EulSafe[o&3];j=EulNext[i+n];k=EulNext[i+1-n];}

	#define EulOrd(i,p,r,f)	   (((((((i)<<1)+(p))<<1)+(r))<<1)+(f))

	#define EulOrdXYZs    EulOrd(0,0,0,0)
	#define EulOrdXYXs    EulOrd(0,0,1,0)
	#define EulOrdXZYs    EulOrd(0,1,0,0)
	#define EulOrdXZXs    EulOrd(0,1,1,0)
	#define EulOrdYZXs    EulOrd(1,0,0,0)
	#define EulOrdYZYs    EulOrd(1,0,1,0)
	#define EulOrdYXZs    EulOrd(1,1,0,0)
	#define EulOrdYXYs    EulOrd(1,1,1,0)
	#define EulOrdZXYs    EulOrd(2,0,0,0)
	#define EulOrdZXZs    EulOrd(2,0,1,0)
	#define EulOrdZYXs    EulOrd(2,1,0,0)
	#define EulOrdZYZs    EulOrd(2,1,1,0)

	#define EulOrdZYXr    EulOrd(0,0,0,1)
	#define EulOrdXYXr    EulOrd(0,0,1,1)
	#define EulOrdYZXr    EulOrd(0,1,0,1)
	#define EulOrdXZXr    EulOrd(0,1,1,1)
	#define EulOrdXZYr    EulOrd(1,0,0,1)
	#define EulOrdYZYr    EulOrd(1,0,1,1)
	#define EulOrdZXYr    EulOrd(1,1,0,1)
	#define EulOrdYXYr    EulOrd(1,1,1,1)
	#define EulOrdYXZr    EulOrd(2,0,0,1)
	#define EulOrdZXZr    EulOrd(2,0,1,1)
	#define EulOrdXYZr    EulOrd(2,1,0,1)
	#define EulOrdZYZr    EulOrd(2,1,1,1)

	int order;
	switch ( euler )
	{
		case EULER_XYZ: order = EulOrdXYZs; break;
		case EULER_XZY: order = EulOrdXZYs; break;
		case EULER_YXZ: order = EulOrdYXZs; break;
		case EULER_YZX: order = EulOrdYZXs; break;
		case EULER_ZXY: order = EulOrdZXYs; break;
		case EULER_ZYX: order = EulOrdZYXs; break;
	}

	int i,j,k,n,s,f;
	EulGetOrd(order,i,j,k,n,s,f);
	Vector3 v;

	if ( s == 1 )
	{
		float sy = (float)sqrt( m44[j][i]*m44[j][i] + m44[k][i]*m44[k][i] );
		if ( sy > FLT_EPSILON*16 )
		{
			v.x = (float)atan2( m44[j][i],m44[k][i] );
			v.y = (float)atan2( sy,m44[i][i] );
			v.z = (float)atan2( m44[i][j],-m44[i][k] );
		}
		else
		{
			v.x = (float)atan2( -m44[k][j],m44[j][j] );
			v.y = (float)atan2( sy,m44[i][i] );
			v.z = 0;
		}
	}
	else
	{
		float cy = (float)sqrt( m44[i][i]*m44[i][i] + m44[i][j]*m44[i][j] );
		if ( cy > FLT_EPSILON*16 )
		{
			v.x = (float)atan2( m44[j][k],m44[k][k] );
			v.y = (float)atan2( -m44[i][k],cy );
			v.z = (float)atan2( m44[i][j],m44[i][i] );
		} 
		else 
		{
			v.x = (float)atan2( -m44[k][j],m44[j][j] );
			v.y = (float)atan2( -m44[i][k],cy);
			v.z = 0;
		}
	}
	
    if ( n==1 ) { v = -v; }
    if ( f==1 ) { float t = v.x; v.x = v.z;	v.z = t; }

	return v;
}


void Matrix4x4::Identity()
{
	m44[0][0] = 1; m44[0][1] = 0; m44[0][2] = 0; m44[0][3] = 0;
	m44[1][0] = 0; m44[1][1] = 1; m44[1][2] = 0; m44[1][3] = 0;
	m44[2][0] = 0; m44[2][1] = 0; m44[2][2] = 1; m44[2][3] = 0;
	m44[3][0] = 0; m44[3][1] = 0; m44[3][2] = 0; m44[3][3] = 1;
}


void Matrix4x4::Scale(float x, float y, float z)
{
	m44[0][0] = x; m44[0][1] = 0; m44[0][2] = 0; m44[0][3] = 0;
	m44[1][0] = 0; m44[1][1] = y; m44[1][2] = 0; m44[1][3] = 0;
	m44[2][0] = 0; m44[2][1] = 0; m44[2][2] = z; m44[2][3] = 0;
	m44[3][0] = 0; m44[3][1] = 0; m44[3][2] = 0; m44[3][3] = 1;
}


void Matrix4x4::Translate(float x, float y, float z)
{
	m44[0][0] = 1; m44[0][1] = 0; m44[0][2] = 0; m44[0][3] = 0;
	m44[1][0] = 0; m44[1][1] = 1; m44[1][2] = 0; m44[1][3] = 0;
	m44[2][0] = 0; m44[2][1] = 0; m44[2][2] = 1; m44[2][3] = 0;
	m44[3][0] = x; m44[3][1] = y; m44[3][2] = z; m44[3][3] = 1;
}

void Matrix4x4::RotateX(float angle)
{
	float s = (float)sin( angle );
	float c = (float)cos( angle );
	
	m44[0][0] = 1; m44[0][1] = 0; m44[0][2] = 0; m44[0][3] = 0;
	m44[1][0] = 0; m44[1][1] = c; m44[1][2] = s; m44[1][3] = 0;
	m44[2][0] = 0; m44[2][1] =-s; m44[2][2] = c; m44[2][3] = 0;
	m44[3][0] = 0; m44[3][1] = 0; m44[3][2] = 0; m44[3][3] = 1;
}


void Matrix4x4::RotateY(float angle)
{
	float s = (float)sin( angle );
	float c = (float)cos( angle );
	
	m44[0][0] = c; m44[0][1] = 0; m44[0][2] =-s; m44[0][3] = 0;
	m44[1][0] = 0; m44[1][1] = 1; m44[1][2] = 0; m44[1][3] = 0;
	m44[2][0] = s; m44[2][1] = 0; m44[2][2] = c; m44[2][3] = 0;
	m44[3][0] = 0; m44[3][1] = 0; m44[3][2] = 0; m44[3][3] = 1;
}


void Matrix4x4::RotateZ(float angle)
{
	float s = (float)sin( angle );
	float c = (float)cos( angle );
	
	m44[0][0] = c; m44[0][1] = s; m44[0][2] = 0; m44[0][3] = 0;
	m44[1][0] =-s; m44[1][1] = c; m44[1][2] = 0; m44[1][3] = 0;
	m44[2][0] = 0; m44[2][1] = 0; m44[2][2] = 1; m44[2][3] = 0;
	m44[3][0] = 0; m44[3][1] = 0; m44[3][2] = 0; m44[3][3] = 1;
}


void Matrix4x4::RotateXYZ(float yaw, float pitch, float roll)
{
	float sy = (float)sin( yaw );
	float cy = (float)cos( yaw );
	float sp = (float)sin( pitch );
	float cp = (float)cos( pitch );
	float sr = (float)sin( roll );
	float cr = (float)cos( roll );
	float spsy = sp * sy;
	float spcy = sp * cy;

	m44[0][0] = cr * cp;
	m44[0][1] = sr * cp;
	m44[0][2] = -sp;
	m44[0][3] = 0;
	m44[1][0] = cr * spsy - sr * cy;
	m44[1][1] = sr * spsy + cr * cy;
	m44[1][2] = cp * sy;
	m44[1][3] = 0;
	m44[2][0] = cr * spcy + sr * sy;
	m44[2][1] = sr * spcy - cr * sy;
	m44[2][2] = cp * cy;
	m44[2][3] = 0;
	m44[3][0] = 0;
	m44[3][1] = 0;
	m44[3][2] = 0;
	m44[3][3] = 1;
}


void Matrix4x4::RotateEuler(const Vector3& v, EulerOrder euler)
{
	switch ( euler )
	{
		case EULER_XYZ: RotateX(v.x); MultRotateY(v.y); MultRotateZ(v.z); break;
		case EULER_XZY: RotateX(v.x); MultRotateZ(v.z); MultRotateY(v.y); break;
		case EULER_YXZ: RotateY(v.y); MultRotateX(v.x); MultRotateZ(v.z); break;
		case EULER_YZX: RotateY(v.y); MultRotateZ(v.z); MultRotateX(v.x); break;
		case EULER_ZXY: RotateZ(v.z); MultRotateX(v.x); MultRotateY(v.y); break;
		case EULER_ZYX: RotateZ(v.z); MultRotateY(v.y); MultRotateX(v.x); break;
		default: break;
	}
}


void Matrix4x4::RotateAngleAxis(float angle, const Vector3& axis)
{
	if ( !LengthSquared(axis) )
	{
		Identity();
		return;
	}
	
	Vector3 n = Normalize(axis);
	float s = (float)sin( angle );
	float c = (float)cos( angle );
	float k = 1 - c;

	float xx = n.x * n.x * k + c;
	float yy = n.y * n.y * k + c;
	float zz = n.z * n.z * k + c;
	float xy = n.x * n.y * k;
	float yz = n.y * n.z * k;
	float zx = n.z * n.x * k;
	float xs = n.x * s;
	float ys = n.y * s;
	float zs = n.z * s;
	
	m44[0][0] = xx;
	m44[0][1] = xy + zs;
	m44[0][2] = zx - ys;
	m44[0][3] = 0;
	m44[1][0] = xy - zs;
	m44[1][1] = yy;
	m44[1][2] = yz + xs;
	m44[1][3] = 0;
	m44[2][0] = zx + ys;
	m44[2][1] = yz - xs;
	m44[2][2] = zz;
	m44[2][3] = 0;
	m44[3][0] = 0;
	m44[3][1] = 0;
	m44[3][2] = 0;
	m44[3][3] = 1;
}


void Matrix4x4::MultScale(float sx, float sy, float sz)
{
	for ( int i=0; i<4; i++ )
	{
		m44[i][0] *= sx;
		m44[i][1] *= sy;
		m44[i][2] *= sz;
	}
}


void Matrix4x4::MultTranslate(float tx, float ty, float tz)
{
	for ( int i=0; i<4; i++ )
	{
		float& e = m44[i][3];
		m44[i][0] += e * tx;
		m44[i][1] += e * ty;
		m44[i][2] += e * tz;
	}
}


void Matrix4x4::MultRotateX(float angle)
{
	float s = (float)sin( angle );
	float c = (float)cos( angle );
	
	for ( int i=0; i<4; i++ )
	{
		float y = m44[i][1];
		float z = m44[i][2];
		m44[i][1] = y * c - z * s;
		m44[i][2] = z * c + y * s;
	}
}


void Matrix4x4::MultRotateY(float angle)
{
	float s = (float)sin( angle );
	float c = (float)cos( angle );
	
	for ( int i=0; i<4; i++ )
	{
		float x = m44[i][0];
		float z = m44[i][2];
		m44[i][0] = x * c + z * s;
		m44[i][2] = z * c - x * s;
	}
}


void Matrix4x4::MultRotateZ(float angle)
{
	float s = (float)sin( angle );
	float c = (float)cos( angle );
	
	for ( int i=0; i<4; i++ )
	{
		float x = m44[i][0];
		float y = m44[i][1];
		m44[i][0] = x * c - y * s;
		m44[i][1] = x * s + y * c;
	}
}


void Matrix4x4::MultRotateXYZ(float yaw, float pitch, float roll)
{
	float sy = (float)sin( yaw );
	float cy = (float)cos( yaw );
	float sp = (float)sin( pitch );
	float cp = (float)cos( pitch );
	float sr = (float)sin( roll );
	float cr = (float)cos( roll );

	float m00 = cr * cp;
	float m01 = sr * cp;
	float m02 = -sp;
	float m10 = sp * sy - sr * cy;
	float m11 = sr * sp * sy + cr * cy;
	float m12 = cp * sy;
	float m20 = cr * sp * cy + sr * sy;
	float m21 = sr * sp * cy - cr * sy;
	float m22 = cp * cy;

	for ( int i=0; i<4; i++ )
	{
		float x = m44[i][0];
		float y = m44[i][1];
		float z = m44[i][2];
		m44[i][0] = x * m00 + y * m10 + z * m20;
		m44[i][1] = x * m01 + y * m11 + z * m21;
		m44[i][2] = x * m02 + y * m12 + z * m22;
	}
}


void Matrix4x4::MultRotateEuler(const Vector3& v, EulerOrder euler)
{
	switch ( euler )
	{
		case EULER_XYZ: MultRotateX(v.x); MultRotateY(v.y); MultRotateZ(v.z); break;
		case EULER_XZY: MultRotateX(v.x); MultRotateZ(v.z); MultRotateY(v.y); break;
		case EULER_YXZ: MultRotateY(v.y); MultRotateX(v.x); MultRotateZ(v.z); break;
		case EULER_YZX: MultRotateY(v.y); MultRotateZ(v.z); MultRotateX(v.x); break;
		case EULER_ZXY: MultRotateZ(v.z); MultRotateX(v.x); MultRotateY(v.y); break;
		case EULER_ZYX: MultRotateZ(v.z); MultRotateY(v.y); MultRotateX(v.x); break;
		default: break;
	}
}


void Matrix4x4::MultMatrix4x4(const Matrix4x4& right)
{
	const mat44& m = right.m44;
	for ( int i=0; i<4; i++ )
	{
		float v[4];
		v[0] = m44[i][0];
		v[1] = m44[i][1];
		v[2] = m44[i][2];
		v[3] = m44[i][3];
		m44[i][0] = v[0]*m[0][0] + v[1]*m[1][0] + v[2]*m[2][0] + v[3]*m[3][0];
		m44[i][1] = v[0]*m[0][1] + v[1]*m[1][1] + v[2]*m[2][1] + v[3]*m[3][1];
		m44[i][2] = v[0]*m[0][2] + v[1]*m[1][2] + v[2]*m[2][2] + v[3]*m[3][2];
		m44[i][3] = v[0]*m[0][3] + v[1]*m[1][3] + v[2]*m[2][3] + v[3]*m[3][3];
	}
}


void Matrix4x4::MultMatrix3x4(const Matrix4x4& right)
{
	const mat44& m = right.m44;
	for ( int i=0; i<4; i++ )
	{
		float v[3];
		v[0] = m44[i][0];
		v[1] = m44[i][1];
		v[2] = m44[i][2];
		m44[i][0] = v[0]*m[0][0] + v[1]*m[1][0] + v[2]*m[2][0];
		m44[i][1] = v[0]*m[0][1] + v[1]*m[1][1] + v[2]*m[2][1];
		m44[i][2] = v[0]*m[0][2] + v[1]*m[1][2] + v[2]*m[2][2];
	}
	m44[3][0] += m[3][0];
	m44[3][1] += m[3][1];
	m44[3][2] += m[3][2];
}


void Matrix4x4::MultMatrix3x3(const Matrix4x4& right)
{
	const mat44& m = right.m44;
	for ( int i=0; i<4; i++ )
	{
		float v[3];
		v[0] = m44[i][0];
		v[1] = m44[i][1];
		v[2] = m44[i][2];
		m44[i][0] = v[0]*m[0][0] + v[1]*m[1][0] + v[2]*m[2][0];
		m44[i][1] = v[0]*m[0][1] + v[1]*m[1][1] + v[2]*m[2][1];
		m44[i][2] = v[0]*m[0][2] + v[1]*m[1][2] + v[2]*m[2][2];
	}
}


void Matrix4x4::MultInverseMatrix3x3(const Matrix4x4& right)
{
	float m00 = m44[0][0];
	float m01 = m44[0][1];
	float m02 = m44[0][2];
	float m10 = m44[1][0];
	float m11 = m44[1][1];
	float m12 = m44[1][2];
	float m20 = m44[2][0];
	float m21 = m44[2][1];
	float m22 = m44[2][2];

	for ( int i=0; i<4; i++ )
	{
		const float* v = right.m16 + i*4;
		m44[i][0] = v[0]*m00 + v[1]*m10 + v[2]*m20;
		m44[i][1] = v[0]*m01 + v[1]*m11 + v[2]*m21;
		m44[i][2] = v[0]*m02 + v[1]*m12 + v[2]*m22;
	}
}


void Matrix4x4::Transpose()
{
	Matrix4x4 v;

	v.m44[0][0] = m44[0][0];
	v.m44[0][1] = m44[1][0];
	v.m44[0][2] = m44[2][0];
	v.m44[0][3] = m44[3][0];
	v.m44[1][0] = m44[0][1];
	v.m44[1][1] = m44[1][1];
	v.m44[1][2] = m44[2][1];
	v.m44[1][3] = m44[3][1];
	v.m44[2][0] = m44[0][2];
	v.m44[2][1] = m44[1][2];
	v.m44[2][2] = m44[2][2];
	v.m44[2][3] = m44[3][2];
	v.m44[3][0] = m44[0][3];
	v.m44[3][1] = m44[1][3];
	v.m44[3][2] = m44[2][3];
	v.m44[3][3] = m44[3][3];

	*this = v;
}


void Matrix4x4::Transpose(const Matrix4x4& m)
{
	m44[0][0] = m.m44[0][0];
	m44[0][1] = m.m44[1][0];
	m44[0][2] = m.m44[2][0];
	m44[0][3] = m.m44[3][0];
	m44[1][0] = m.m44[0][1];
	m44[1][1] = m.m44[1][1];
	m44[1][2] = m.m44[2][1];
	m44[1][3] = m.m44[3][1];
	m44[2][0] = m.m44[0][2];
	m44[2][1] = m.m44[1][2];
	m44[2][2] = m.m44[2][2];
	m44[2][3] = m.m44[3][2];
	m44[3][0] = m.m44[0][3];
	m44[3][1] = m.m44[1][3];
	m44[3][2] = m.m44[2][3];
	m44[3][3] = m.m44[3][3];
}


void Matrix4x4::Adjoint()
{
	mat44& m = m44;
	Matrix4x4 v;

	v.m44[0][0] = m[1][1]*m[2][2] - m[1][2]*m[2][1];
	v.m44[0][1] = m[2][1]*m[0][2] - m[2][2]*m[0][1];
	v.m44[0][2] = m[0][1]*m[1][2] - m[0][2]*m[1][1];
	v.m44[0][3] = m[0][3];
	v.m44[1][0] = m[1][2]*m[2][0] - m[1][0]*m[2][2];
	v.m44[1][1] = m[2][2]*m[0][0] - m[2][0]*m[0][2];
	v.m44[1][2] = m[0][2]*m[1][0] - m[0][0]*m[1][2];
	v.m44[1][3] = m[1][3];
	v.m44[2][0] = m[1][0]*m[2][1] - m[1][1]*m[2][0];
	v.m44[2][1] = m[2][0]*m[0][1] - m[2][1]*m[0][0];
	v.m44[2][2] = m[0][0]*m[1][1] - m[0][1]*m[1][0];
	v.m44[2][3] = m[2][3];
	v.m44[3][0] = -(m[0][0]*m[3][0] + m[1][0]*m[3][1] + m[2][0]*m[3][2]);
	v.m44[3][1] = -(m[0][1]*m[3][0] + m[1][1]*m[3][1] + m[2][1]*m[3][2]);
	v.m44[3][2] = -(m[0][2]*m[3][0] + m[1][2]*m[3][1] + m[2][2]*m[3][2]);
	v.m44[3][3] = m[3][3];

	*this = v;
}


void Matrix4x4::Adjoint(const Matrix4x4& a)
{
	const mat44& m = a.m44;

	m44[0][0] = m[1][1]*m[2][2] - m[1][2]*m[2][1];
	m44[0][1] = m[2][1]*m[0][2] - m[2][2]*m[0][1];
	m44[0][2] = m[0][1]*m[1][2] - m[0][2]*m[1][1];
	m44[0][3] = m[0][3];
	m44[1][0] = m[1][2]*m[2][0] - m[1][0]*m[2][2];
	m44[1][1] = m[2][2]*m[0][0] - m[2][0]*m[0][2];
	m44[1][2] = m[0][2]*m[1][0] - m[0][0]*m[1][2];
	m44[1][3] = m[1][3];
	m44[2][0] = m[1][0]*m[2][1] - m[1][1]*m[2][0];
	m44[2][1] = m[2][0]*m[0][1] - m[2][1]*m[0][0];
	m44[2][2] = m[0][0]*m[1][1] - m[0][1]*m[1][0];
	m44[2][3] = m[2][3];
	m44[3][0] = -(m[0][0]*m[3][0] + m[1][0]*m[3][1] + m[2][0]*m[3][2]);
	m44[3][1] = -(m[0][1]*m[3][0] + m[1][1]*m[3][1] + m[2][1]*m[3][2]);
	m44[3][2] = -(m[0][2]*m[3][0] + m[1][2]*m[3][1] + m[2][2]*m[3][2]);
	m44[3][3] = m[3][3];
}


void Matrix4x4::Inverse()
{
	float s = 1 / GetDeterminant();
	mat44& m = m44;
	Matrix4x4 v;

	v.m44[0][0] = (m[1][1]*m[2][2] - m[1][2]*m[2][1]) * s;
	v.m44[0][1] = (m[2][1]*m[0][2] - m[2][2]*m[0][1]) * s;
	v.m44[0][2] = (m[0][1]*m[1][2] - m[0][2]*m[1][1]) * s;
	v.m44[0][3] = m[0][3];
	v.m44[1][0] = (m[1][2]*m[2][0] - m[1][0]*m[2][2]) * s;
	v.m44[1][1] = (m[2][2]*m[0][0] - m[2][0]*m[0][2]) * s;
	v.m44[1][2] = (m[0][2]*m[1][0] - m[0][0]*m[1][2]) * s;
	v.m44[1][3] = m[1][3];
	v.m44[2][0] = (m[1][0]*m[2][1] - m[1][1]*m[2][0]) * s;
	v.m44[2][1] = (m[2][0]*m[0][1] - m[2][1]*m[0][0]) * s;
	v.m44[2][2] = (m[0][0]*m[1][1] - m[0][1]*m[1][0]) * s;
	v.m44[2][3] = m[2][3];
	v.m44[3][0] =-(v.m44[0][0]*m[3][0] + v.m44[1][0]*m[3][1] + v.m44[2][0]*m[3][2]);
	v.m44[3][1] =-(v.m44[0][1]*m[3][0] + v.m44[1][1]*m[3][1] + v.m44[2][1]*m[3][2]);
	v.m44[3][2] =-(v.m44[0][2]*m[3][0] + v.m44[1][2]*m[3][1] + v.m44[2][2]*m[3][2]);
	v.m44[3][3] = m[3][3];

	*this = v;
}


void Matrix4x4::Inverse(const Matrix4x4& i)
{
	float s = 1 / i.GetDeterminant();
	const mat44& m = i.m44;

	m44[0][0] = (m[1][1]*m[2][2] - m[1][2]*m[2][1]) * s;
	m44[0][1] = (m[2][1]*m[0][2] - m[2][2]*m[0][1]) * s;
	m44[0][2] = (m[0][1]*m[1][2] - m[0][2]*m[1][1]) * s;
	m44[0][3] = m[0][3];
	m44[1][0] = (m[1][2]*m[2][0] - m[1][0]*m[2][2]) * s;
	m44[1][1] = (m[2][2]*m[0][0] - m[2][0]*m[0][2]) * s;
	m44[1][2] = (m[0][2]*m[1][0] - m[0][0]*m[1][2]) * s;
	m44[1][3] = m[1][3];
	m44[2][0] = (m[1][0]*m[2][1] - m[1][1]*m[2][0]) * s;
	m44[2][1] = (m[2][0]*m[0][1] - m[2][1]*m[0][0]) * s;
	m44[2][2] = (m[0][0]*m[1][1] - m[0][1]*m[1][0]) * s;
	m44[2][3] = m[2][3];
	m44[3][0] =-(m44[0][0]*m[3][0] + m44[1][0]*m[3][1] + m44[2][0]*m[3][2]);
	m44[3][1] =-(m44[0][1]*m[3][0] + m44[1][1]*m[3][1] + m44[2][1]*m[3][2]);
	m44[3][2] =-(m44[0][2]*m[3][0] + m44[1][2]*m[3][1] + m44[2][2]*m[3][2]);
	m44[3][3] = m[3][3];
}


void Matrix4x4::Inverse(const Quaternion& q)
{
	float xs = q.x * 2;
	float ys = q.y * 2;
	float zs = q.z * 2;
	float wx = q.w * xs;
	float wy = q.w * ys;
	float wz = q.w * zs;
	float xx = q.x * xs;
	float xy = q.x * ys;
	float xz = q.x * zs;
	float yy = q.y * ys;
	float yz = q.y * zs;
	float zz = q.z * zs;

	m44[0][0] = 1 - (yy + zz);
	m44[0][1] = xy + wz;
	m44[0][2] = xz - wy;
	m44[0][3] = 0;
	m44[1][0] = xy - wz;
	m44[1][1] = 1 - (xx + zz);
	m44[1][2] = yz + wx;
	m44[1][3] = 0;
	m44[2][0] = xz + wy;
	m44[2][1] = yz - wx;
	m44[2][2] = 1 - (xx + yy);
	m44[2][3] = 0;
	m44[3][0] = 0;
	m44[3][1] = 0;
	m44[3][2] = 0;
	m44[3][3] = 1;
}


void Matrix4x4::OrthoNormalize()
{
	Vector3 x = GetX();
	Vector3 y = GetY();
	Vector3 z = GetZ();

	x = Normalize(x);
	y -= x * DotProduct(x,y);
	y = Normalize(y);
	z = CrossProduct(x,y);

	SetX( x );
	SetY( y );
	SetZ( z );
}


void Matrix4x4::Mirror(const Matrix4x4& transform, const Plane& plane)
{
	// components
	Vector3 x = transform.GetX();
	Vector3 y = transform.GetY();
	Vector3 z = transform.GetZ();
	Vector3 t = transform.GetT();
	Vector3 n2 = plane.normal * -2;

	// mirror translation
	Vector3 mt = t + n2 * (DotProduct(t,plane.normal) - plane.dist);
	
	// mirror x rotation
	x += t;
	x += n2 * (DotProduct(x,plane.normal) - plane.dist);
	x -= mt;

	// mirror y rotation
	y += t;
	y += n2 * (DotProduct(y,plane.normal) - plane.dist);
	y -= mt;

	// mirror z rotation
	z += t;
	z += n2 * (DotProduct(z,plane.normal) - plane.dist);
	z -= mt;

	// write result
	SetX( x );
	SetY( y );
	SetZ( z );
	SetT( mt );
	m44[0][3] = 0; 
	m44[1][3] = 0; 
	m44[2][3] = 0; 
	m44[3][3] = 1;
}


void Matrix4x4::LookAt(const Vector3& target, const Vector3& view, const Vector3& up)
{
	Vector3 z = Normalize(target - view);
	Vector3 x = Normalize(CrossProduct(up,z));
	Vector3 y = CrossProduct(z,x);

    m44[0][0] = x.x;
    m44[1][0] = x.y;
    m44[2][0] = x.z;
    m44[3][0] = -DotProduct(x,view);
    m44[0][1] = y.x;
    m44[1][1] = y.y;
    m44[2][1] = y.z;
    m44[3][1] = -DotProduct(y,view);
    m44[0][2] = z.x;
    m44[1][2] = z.y;
    m44[2][2] = z.z;
    m44[3][2] = -DotProduct(z,view);
    m44[0][3] = 0;
    m44[1][3] = 0;
    m44[2][3] = 0;
    m44[3][3] = 1;
}
