#include <windows.h>
#include <gl\gl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "mathlib.h"
#include "heightfield.h"

#pragma warning(disable : 4244)

heightfield_t hfield;

static const float ambient = 0.05;
static const float reflect = 0.6;

int HField_Init(int width, int height, int spacing)
{
	hfield.age = 0.0;
	hfield.effect = hfield.nexteffect = NONE;
	hfield.texnum = 1;
	hfield.width = width;
	hfield.height = height;
	hfield.spacing = spacing;
	hfield.numverts = (width + 1) * (height + 1);
	hfield.numfaces = width * height * 2;
	hfield.verts = (void*)hfield.vnorms = (void*)hfield.tcoords = (void*)hfield.vcolors = (void*)hfield.faces = (void*)hfield.fnorms = NULL;
	if ( !(hfield.verts = (vec3_t*)malloc(hfield.numverts * sizeof(vec3_t))) ) return 0;
	if ( !(hfield.vdists = (float*)malloc(hfield.numverts * sizeof(float))) ) return 0;
	if ( !(hfield.vsins = (float*)malloc(hfield.numverts * sizeof(float))) ) return 0;
	if ( !(hfield.vnorms = (vec3_t*)malloc(hfield.numverts * sizeof(vec3_t))) ) return HField_Destroy();
	if ( !(hfield.tcoords = (tcoord_t*)malloc(hfield.numverts * sizeof(tcoord_t))) ) return HField_Destroy();
	if ( !(hfield.vcolors = (vec3_t*)malloc(hfield.numverts * sizeof(vec3_t))) ) return HField_Destroy();
	if ( !(hfield.faces = (face_t*)malloc(hfield.numfaces * sizeof(face_t))) ) return HField_Destroy();
	if ( !(hfield.fnorms = (vec3_t*)malloc(hfield.numfaces * sizeof(vec3_t))) ) return HField_Destroy();
	HField_GenVerts();
	HField_GenDists(1.0);
	HField_GenSins(1.0);
	HField_GenFaces();
	HField_GenFaceNormals();
	HField_GenVertNormals();
	HField_GenVertColors();
}

int HField_Destroy(void)
{	if ( hfield.verts ) free(hfield.verts);
	if ( hfield.vnorms ) free(hfield.vnorms);
	if ( hfield.tcoords ) free(hfield.tcoords);
	if ( hfield.vcolors ) free(hfield.vcolors);
	if ( hfield.faces ) free(hfield.faces);
	if ( hfield.fnorms ) free(hfield.fnorms);
	if ( hfield.vnorms ) free(hfield.vnorms);
	hfield.verts = (void*)hfield.vnorms = (void*)hfield.tcoords = (void*)hfield.vcolors = (void*)hfield.faces = (void*)hfield.fnorms = NULL;
	return 0;
}

int HField_GenVerts(void)
{	int row, rowoff, col;
	float x, y;

	y = 0.0 - (float)(hfield.height/2 * hfield.spacing);
	rowoff = 0;
	for ( row = 0; row <= hfield.height; row++ )
	{	x = 0.0 - (float)(hfield.width/2 * hfield.spacing);
		for ( col = 0; col <= hfield.width; col++ )
		{	M_VectSet(hfield.verts[rowoff + col], x, y, rand() % 60 - 30);//
			x += (float)hfield.spacing;
		}
		rowoff += hfield.width + 1;
		y += (float)hfield.spacing;
	}
	return 1;
}

int HField_GenDists(float scale)
{	int i;

	for ( i=0; i<hfield.numverts; i++ )
	{	hfield.vdists[i] = scale * sqrt( hfield.verts[i][0]*hfield.verts[i][0] + hfield.verts[i][1] * hfield.verts[i][1]);
	}

	return 1;
}

int HField_GenSins(float scale)
{	int i;

	for ( i=0; i<hfield.numverts; i++ )
	{	hfield.vsins[i] = sin(hfield.vdists[i]*scale);
	}

	return 1;
}

int HField_GenFaces(void)
{	int row, rowoff, col, face;

	rowoff = 0;
	face = 0;
	for ( row = 0; row < hfield.height; row++ )
	{	for ( col = 0; col < hfield.width; col++ )
		{
			hfield.faces[face][0] = rowoff + col;
			hfield.faces[face][1] = rowoff + col + 1;
			hfield.faces[face++][2] = rowoff + hfield.width + 1 + col;

			hfield.faces[face][0] = rowoff + hfield.width + 1 + col;
			hfield.faces[face][1] = rowoff + col + 1;
			hfield.faces[face++][2] = rowoff + hfield.width + 1 + col + 1;
		}
		rowoff += hfield.width + 1;
	}
	return 1;
}

void HField_Settle(void)
{
	hfield.effect = SETTLE;
}

void HField_StartEffect(hfieldeffect effect)
{	HField_Settle();
	hfield.nexteffect = effect;
}

void HField_InitEffect(hfieldeffect effect)
{	switch ( effect )
	{	case NONE: break;
		case SETTLE: break;
		case WAVY1:
			HField_GenDists(1.0/WAVELENGTH_1);
			HField_GenSins(3.0);
			break;
		default: break;
	}
}

void HField_Animate(void)
{
	switch ( hfield.effect )
	{	case NONE: break;
		case SETTLE: HField_AnimateSettle(); break;
		case WAVY1: HField_AnimateWavy1(); break;
		default: break;
	}
}

void HField_AnimateSettle(void)
{	int i, settled = 0;
	float delta;

	for ( i=0; i<hfield.numverts; i++ )
	{	if ( hfield.verts[i][2] > -SETTLE_THRESHOLD && hfield.verts[i][2] < SETTLE_THRESHOLD )
		{	hfield.verts[i][2] = 0.0;
			settled++;
		}
		else
		{	delta = hfield.verts[i][2] * SETTLE_FACTOR;
			if ( delta > -SETTLE_SPEED && delta < SETTLE_SPEED )
			{	delta = delta > 0.0 ? SETTLE_SPEED : -SETTLE_SPEED;
			}
			hfield.verts[i][2]-= delta * framedelta;
		}
	}
	if ( settled == hfield.numverts )
	{	hfield.effect = hfield.nexteffect;
		HField_InitEffect(hfield.effect);
		hfield.nexteffect = NONE;
	}
	HField_GenFaceNormals();
	HField_GenVertNormals();
	HField_GenVertColors();
}

void HField_AnimateWavy1(void)
{	int i;
	static float lfreq = 0.0;

	lfreq += framedelta * FREQUENCY_1;
	for ( i=0; i<hfield.numverts; i++ )
	{	hfield.verts[i][2] = AMPLITUDE_1* cos(hfield.vdists[i] + lfreq);
	}
	HField_GenFaceNormals();
	HField_GenVertNormals();
	HField_GenVertColors();
}


void HField_Draw(void)
{
	float alpha = .6;
	int row, col, off1, off2;
	vec3_t t;

	glEnable(GL_TEXTURE_2D);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glBindTexture(GL_TEXTURE_2D, HEIGHTFIELD_TEXTURE);
	glEnable(GL_CULL_FACE);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	off1 = 0;
	off2 = hfield.width + 1;
	for ( row=0; row < hfield.height; row++ )
	{
		glBegin(GL_TRIANGLE_STRIP);
		for ( col = 0; col <= hfield.width; col++ )
		{	
			glColor3fv( hfield.vcolors[ off2 ] );
/*			glColor4f(	hfield.vcolors[off2][0],
							hfield.vcolors[off2][1],
							hfield.vcolors[off2][2], alpha );
*/
			M_VectScale(hfield.verts[ off2 ], 1.0/(float)hfield.spacing/4.0, t);
			glTexCoord2fv( t );
			glVertex3fv( hfield.verts[ off2 ] );

			glColor3fv( hfield.vcolors[ off1 ] );
/*			glColor4f(	hfield.vcolors[off1][0],
							hfield.vcolors[off1][1],
							hfield.vcolors[off1][2], alpha );
*/
			M_VectScale(hfield.verts[ off1 ], 1.0/(float)hfield.spacing/4.0, t);
			glTexCoord2fv( t );
			glVertex3fv( hfield.verts[ off1 ] );

			off1++;
			off2++;
		}
		glEnd();
	}
}


void HField_GenVertColors(void)
{	int i;
	float c0[3]={ambient, ambient, ambient};
	float c1[3]={reflect, reflect, reflect};
	float c2[3];
	float dot;

	for ( i=0; i<hfield.numverts; i++ )
	{	dot = M_DotProd(lightvect, hfield.vnorms[i]);
		if ( dot<0.0 ) dot = 0.0;
		M_VectMultAdd(c0, c1, dot, c2);
		M_VectCpy(c2, hfield.vcolors[i]);
	}
}

void HField_GenFaceNormals(void)
{
	int i;
	vec3_t v1, v2, v3;

	for ( i=0; i<hfield.numfaces; i++ )
	{	M_VectSub(hfield.verts[hfield.faces[i][1]], hfield.verts[hfield.faces[i][0]], v1);
		M_VectSub(hfield.verts[hfield.faces[i][2]], hfield.verts[hfield.faces[i][0]], v2);
		M_CrossProd(v1, v2, v3);
		M_VectNormalize(v3, v3);
		M_VectCpy(v3, hfield.fnorms[i]);
	}
}


void HField_GenVertNormals(void)
{
	int *counts;
	int i;

	if ( !(counts = (int*)malloc(hfield.numverts*sizeof(int))) )
	{	exit(-1);
	}
	memset(counts, 0, hfield.numverts*sizeof(int));
	for ( i=0; i< hfield.numverts; i++ ) M_VectSet(hfield.vnorms[i], 0.0, 0.0, 0.0);

	/* sum up face normals for each vertex */
	for ( i=0; i<hfield.numfaces; i++ )
	{	M_VectAdd(hfield.fnorms[i], hfield.vnorms[hfield.faces[i][0]], hfield.vnorms[hfield.faces[i][0]]);
		counts[hfield.faces[i][0]]++;
		M_VectAdd(hfield.fnorms[i], hfield.vnorms[hfield.faces[i][1]], hfield.vnorms[hfield.faces[i][1]]);
		counts[hfield.faces[i][1]]++;
		M_VectAdd(hfield.fnorms[i], hfield.vnorms[hfield.faces[i][2]], hfield.vnorms[hfield.faces[i][2]]);
		counts[hfield.faces[i][2]]++;
	}
	/* divide by number of normals contributing to each vertex sum */
	for ( i=0; i<hfield.numverts; i++ )
	{	M_VectScale(hfield.vnorms[i], 1.0/(float)counts[i], hfield.vnorms[i]);
	}
	free(counts);
}


#pragma warning(default : 4244)