#include <windows.h>
#include <gl\gl.h>
#include <math.h>
#include "particle.h"
#include "mathlib.h"
#include "glfont.h"

#pragma warning(disable : 4244)


extern vec3_t viewpos;
extern vec3_t viewangle;

psystem_t *psystems = NULL; // linked list of all systems
psystem_t *lastsys = NULL;	// pointer to last system
int usednum = 0;	// incremented each time a new system is created
int psysblendmode = GL_ONE;

int Psys_New(psystype type)
// creates a new system, tags it to the end of the list
// and returns the new system's number
{	
	psystem_t *newsys;

	if ( !(newsys = (psystem_t*)malloc(sizeof(psystem_t))) )
	{	return 0;
	}
	memset(newsys, 0, sizeof(psystem_t*));

	newsys->type = type;
	if ( !Psys_Init(newsys) )
	{	free(newsys);
		return 0;
	}

	if ( psystems == NULL )
	{	psystems = newsys;
		lastsys = newsys;
	}
	else
	{	lastsys->next = newsys;
		newsys->prev = lastsys;
		lastsys = newsys;
	}		
	return newsys->sysnum;
}

particle_t	*Psys_NewParticle(void)
{	int i;
	particle_t *part;

	part = (particle_t*)malloc(sizeof(particle_t));
	if ( part == NULL ) return part;
	M_VectSet(part->pos, 0.0, 0.0, 0.0);
	M_VectSet(part->velocity, 0.0, 0.0, 0.0);
	for ( i=0; i<4; part->color[i++]=1.0 );
	part->age = -1.0;
	return part;
}

void Psys_ResetParticle(particle_t *part)
{	int i;

	M_VectSet(part->pos, 0.0, 0.0, 0.0);
	M_VectSet(part->velocity, 0.0, 0.0, 0.0);
	for ( i=0; i<4; part->color[i++]=1.0 );
	part->age = 0.0;
}


void Psys_CompactParticles(psystem_t *psys)
// moves the particles to the front of the array
{	int i, j;
	particle_t *t;

	i = 0;
	j = psys->maxparticles-1;
	while ( i<j )
	{	while ( i<psys->numparticles && psys->particles[i]->age > 0.0 )
		{	i++;
		}
		while ( j>0 && psys->particles[j]->age < 0.0 )
		{	j--;
		}
		if ( i<j )
		{	t = psys->particles[i];
			psys->particles[i] = psys->particles[j];
			psys->particles[j] = t;
			i++;
			j--;
		}
	}
}

psystem_t *Psys_SysForNum(int sysnum)
// returns pointer to the particle system
// with specified number.  NULL if it doesn't exist
{	psystem_t *sys;

	for ( sys=psystems; sys; sys=sys->next )
	{	if ( sys->sysnum == sysnum ) return sys;
	}
	return NULL;
}

int Psys_Init(psystem_t *psys)
// initalizes a system
{
	int i;

	psys->sysnum = ++usednum;
	psys->prev = NULL;
	psys->next = NULL;
	M_VectSet(psys->pos, 0.0, 0.0, 0.0);
	M_VectSet(psys->velocity, 0.0, 0.0, 0.0);
	switch ( psys->type )
	{	case NONE:	return 0;
		case FOUNTAIN:
			psys->running = 1;
			psys->numparticles = 0;
			psys->maxparticles = NUM_FOUNTAIN_PARTICLES;
			psys->age = 0;
			psys->gentime = 0.0;
			psys->gendelta = 0.009;
			psys->nextgen = 0.0;
			break;
		case GRAVFIELD:
			psys->running = 1;
			psys->numparticles = 0;
			psys->maxparticles = NUM_GRAVFIELD_PARTICLES;
			psys->age = 0;
			psys->gentime = 0.0;
			psys->gendelta = 0.03;
			psys->nextgen = 0.0;
			break;
		default: return 0;
	}
	psys->particles = (particle_t**)malloc( sizeof(particle_t**) * psys->maxparticles);
	memset(psys->particles, 0, sizeof(particle_t**) * psys->numparticles);
	for ( i=0; i<psys->maxparticles; i++ )
	{	psys->particles[i] = Psys_NewParticle();
		if ( !psys->particles[i] )
		{	exit(-1);
		}
	}
	return 1;
}

int Psys_RemoveN(int sysnum, int destroy)
// removes a system from chain by number
{	psystem_t *sys;

	if ( !(sys = Psys_SysForNum(sysnum)) ) return 0;
	return Psys_RemoveP(sys, destroy);
}

int Psys_RemoveP(psystem_t *sys, int destroy)
// removes a system from chain by actual structure
{	int i;

	if ( sys->prev ) sys->prev->next = sys->next;
	if ( sys->next ) sys->next->prev = sys->prev;
	if ( sys == psystems ) psystems = sys->next;
	if ( sys == lastsys ) lastsys = sys->prev;
	if ( !destroy ) return 1;

	for ( i=0; i<sys->numparticles; i++ )
	{	free(sys->particles[i]);
	}
	free(sys);
	return 1;
}

void Psys_AnimateAll(void)
{	psystem_t *sys;

	for ( sys = psystems; sys; sys = sys->next )
	{	if ( sys->running )
		{	M_VectMultAdd(sys->pos, sys->velocity, framedelta, sys->pos);
			switch ( sys->type )
			{	case NONE: break;
				case FOUNTAIN: Psys_AnimateFountain(sys); break;
				case GRAVFIELD: Psys_AnimateGravfield(sys); break;
				default: break;
			}
		}
	}
}

void Psys_AnimateFountain(psystem_t *psys)
{	particle_t *part;
	vec3_t angles = {0.0, 0.0, 0.0};
	float colorstep, colorstepb, gravfdelta;
	int i;

	if ( psys->gentime == 0.0 || psys->age + framedelta < psys->gentime )
	{
		psys->age += framedelta;
		while (	psys->numparticles < psys->maxparticles && psys->age > psys->nextgen )
		{	psys->nextgen += psys->gendelta;
			part = psys->particles[psys->numparticles];
			Psys_ResetParticle(part);
			part->age = 0.0;
			part->color[0] = (float)(rand()%50 + 155)/255;
			part->color[1] = part->color[0];
			part->color[2] = 1.0;
			part->color[3] = 0.4;
			angles[YAW] = (float)(rand() %360);
			angles[PITCH] = (float)(rand()%15 + 75);
			M_AngToVects(angles, part->velocity, NULL, NULL);
			M_VectCpy(part->velocity, part->pos);
			part->pos[2] = 0.0;
			M_VectScale(part->pos, (float)(rand() % 5), part->pos);
			M_VectScale(part->velocity, (float)(rand() % 20 + 170), part->velocity);
//			part->velocity[2] = 20+(float)(rand() % 150);
			psys->numparticles++;
		}
	}

	colorstep = framedelta / 8.0;
	colorstepb = colorstep * 1.2;
	gravfdelta = GRAVITY * framedelta;
	for ( i=0; i<psys->numparticles; i++ )
	{	part = psys->particles[i];
		part->age += framedelta;
		part->color[0] -= colorstep;
		part->color[1] -= colorstep;
		part->color[2] -= colorstepb;
		if ( part->age >= 2.0 )
		{	part->age = -1.0;
			psys->numparticles--;
		}
		else
		{	M_VectMultAdd(part->pos, part->velocity, framedelta, part->pos); // move particle
			part->velocity[2]+=gravfdelta;
		}
	}
	Psys_CompactParticles(psys);
}

void Psys_AnimateGravfield(psystem_t *psys)
{	particle_t *part;
	vec3_t angles = {0.0, 0.0, 0.0};
	vec3_t fdvec;
	static int init = 0;
	static float yangle = 0, pangle = 0;
	float posmult, vscale, colorstepr, colorstepg, colorstepb;
	int i;

	if ( !init )
	{	init = 1;
		srand(timeGetTime());
		yangle = rand() % 72000 - 36000;
		pangle = rand() % 72000 - 36000;
	}

	if ( psys->gentime == 0.0 || psys->age + framedelta < psys->gentime )
	{
		psys->age += framedelta;
		while (	psys->numparticles < psys->maxparticles  && psys->age > psys->nextgen )
		{	psys->nextgen += psys->gendelta;
			part = psys->particles[psys->numparticles];
			Psys_ResetParticle(part);
			part->age = 0.0;
//			part->color[0] = (float)(rand()%100 + 155)/255;
			part->color[0] = 0.5;
			part->color[1] = 1.0;
			part->color[2] = 1.0;
			part->color[3] = 0.7;
			angles[PITCH] = (pangle+=.1);
			angles[YAW] = (yangle+=19);
			M_AngToVects(angles, part->pos, NULL, NULL);
			M_VectScale(part->pos, 50.0 * cos(pangle/100.0), part->pos);
			angles[YAW] += -90;
			angles[PITCH] += 60;
			M_AngToVects(angles, part->velocity, NULL, NULL);
			M_VectScale(part->velocity, (float)10 , part->velocity);
			psys->numparticles++;
			if ( yangle > 36000 ) yangle -= 72000;
			if ( pangle > 36000 ) pangle -= 72000;
		}
	}

	posmult = -.15 * framedelta;
	vscale = 1 + .1*framedelta;
	colorstepr = framedelta / -100.0;
	colorstepg = framedelta / -3.0;
	colorstepb = framedelta / -25.0;

	M_VectSet(fdvec, framedelta, framedelta, framedelta);
	for ( i=0; i<psys->numparticles; i++ )
	{	part = psys->particles[i];
		part->age += framedelta;
		part->color[0] += colorstepr;
		part->color[1] += colorstepg;
		part->color[2] += colorstepb;
		if ( part->age >= 30.0 )
		{	part->age = -1.0;
			psys->numparticles--;
		}
		else
		{	M_VectMultAdd(part->pos, part->velocity, framedelta, part->pos); // move particle
			M_VectMultAdd(part->velocity, part->pos, posmult, part->velocity);
			M_VectScale(part->velocity, vscale, part->velocity);
		}
	}
	Psys_CompactParticles(psys);
}

int Psys_Partition(particle_t **parts, int l, int r, int axis)
{	particle_t *t;
	int i, j;

	// swap first with middle, cuz the middle should make a better pivot
	t = parts[(l+r)/2];
	parts[(l+r)/2] = parts[l];
	parts[l] = t;

	i=l+1;
	for ( i=l+1, j=l; i<=r; i++ )
	{	if ( parts[i]->pos[axis] < parts[l]->pos[axis] )
		{	j++;
			t = parts[j];
			parts[j] = parts[i];
			parts[i] = t;
		}
	}
	t = parts[l];
	parts[l] = parts[j];
	parts[j] = t;
	return j;
}

void Psys_SortParticles(psystem_t *psys, int l, int r, int axis)
// sorts particles for drawing.
{	int q;

	if ( l < r )
	{	q = Psys_Partition(psys->particles, l, r, axis);
		Psys_SortParticles(psys, l, q-1, axis);
		Psys_SortParticles(psys, q+1, r, axis);
	}
}

void Psys_DrawAll(void)
{	psystem_t *sys;

	for ( sys = psystems; sys; sys = sys->next )
	{	Psys_DrawP(sys);
	}
}

void Psys_DrawN(int sysnum)
{	psystem_t *sys;

	if ( sys = Psys_SysForNum(sysnum) )
	{	Psys_DrawP(sys);
	}
}

void Psys_DrawP(psystem_t *psys)
{	particle_t *part;
	vec3_t right, up, offset;
	int i, dir=1, end;
	char str[80];

	M_VectSub(psys->pos, viewpos, offset);
	M_VectCpy(offset, right);
	for ( i=0; i<3; i++ )
	{	 if ( offset[i]<0 ) right[i]=-1.0*offset[i];
	}
//	sprintf(str, "offset = (%f, %f, %f)", offset[0], offset[1], offset[2]);
//	Font_BlitString(str, 0, 10+30*psys->type, 0);
//	sprintf(str, "right = (%f, %f, %f)", right[0], right[1], right[2]);
//	Font_BlitString(str, 0, 20+30*psys->type, 0);
	i=M_VectMax(right);
	dir = offset[i]<=0 ? 1:-1;
	Psys_SortParticles(psys, 0, psys->numparticles-1, i);
//	sprintf(str, "Sorting on %d axis, traversing %d, particles: %d", i, dir, psys->numparticles);
//	Font_BlitString(str, 0, 30+30*psys->type, 0);

	M_AngToVects(viewangle, NULL, right, up);
	M_VectScale(right, PARTICLE_WIDTH, right);
	M_VectScale(up, PARTICLE_WIDTH, up);

	/* set projection matrix for object translation */
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();// save world translation matrix
		glTranslatef(psys->pos[0], psys->pos[1], psys->pos[2]);
		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, PARTICLE_TEXTURE);
		glDisable(GL_CULL_FACE);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, psysblendmode);

		if ( psys->running )
		{
			glBegin(GL_QUADS);
			i=0;
			end = psys->numparticles-1;
			if ( dir==-1 )
			{	i = psys->numparticles-1;
				end = 0;
			}
			
			while ( 1 )
			{	if ( dir == 1 )
				{	 if ( i > end ) break;
				}
				else
				{	if ( i< end ) break;
				}
				part = psys->particles[i];
				glColor4f(part->color[0], part->color[1], part->color[2], part->color[3]);

				M_VectSub(part->pos, right, offset);
				M_VectSub(offset, up, offset);

				glTexCoord2f(0.0, 1.0);
				glVertex3fv(offset);

				M_VectAdd(offset, up, offset);
				M_VectAdd(offset, up, offset);
				glTexCoord2f(0.0, 0.0);
				glVertex3fv(offset);

				M_VectAdd(offset, right, offset);
				M_VectAdd(offset, right, offset);
				glTexCoord2f(1.0, 0.0);
				glVertex3fv(offset);

				M_VectSub(offset, up, offset);
				M_VectSub(offset, up, offset);
				glTexCoord2f(1.0, 1.0);
				glVertex3fv(offset);

				i+=dir;
			}
			glEnd();
		}

	glPopMatrix();	// restore projection matrix
	glDisable(GL_BLEND);
	glEnable(GL_CULL_FACE);
}

void Psys_SetPosN(int sysnum, float x, float y, float z)
{	psystem_t *sys;

	if ( sys = Psys_SysForNum(sysnum) )
	{	Psys_SetPosP(sys, x, y, z);
	}
}
	
void Psys_SetPosP(psystem_t *sys, float x, float y, float z)
{	sys->pos[0] = x;
	sys->pos[1] = y;
	sys->pos[2] = z;
}


#pragma warning(default : 4244)
