/* gltest.c */

#include <windows.h>
#include <time.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <mmsystem.h>
#include <direct.h> // _chdir
#include "gldemo.h"
#include "resource.h"

#pragma warning(disable : 4244)

#define INDEX_UP 0
#define INDEX_DOWN 1
#define INDEX_LEFT 2
#define INDEX_RIGHT 3
#define INDEX_PGUP 4
#define INDEX_PGDN 5
#define INDEX_HOME 6
#define INDEX_END 7
#define INDEX_DELETE 8
#define FRAMETIME .02


LRESULT CALLBACK WindowProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK DlgProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
void init(void);
void redraw(void);
void resize(void);
BOOL setupPixelFormat(HDC hDC);
void UpdateViewPos(void);
void InitTimes(void);
void UpdateTimes(void);

char lpszWinName[] = "MyWin";
char lpszAppName[] = "Numb";

HINSTANCE hInst;	/* this instance of the application, copied to a
							global so it's available to everything */

int winX = 640, winY = 480;
int winWidth = 640, winHeight = 480;
float spinspeed;
vec3_t viewpos = {0.0F, 0.0F, 0.0F};
vec3_t viewangle = {0.0F, 0.0F, 0.0F};
vec3_t viewvect = {0.0, 0.0, 0.0};
//vec3_t lightvect = {.577350, .577350, -.577350};
vec3_t lightvect = {0.0, -1.0, 0.3};
float	framedelta=0.0;
int lasttime=0,
	firstframe=0,
	lastframe=0,
	framenum=0;
BOOL DoneWithMessageLoop = FALSE;

HDC hDC = 0;
HGLRC hGLRC = 0;
HWND	hWindow = 0,
		hDialog = 0;

DEVMODE dmode;
int modenum = 0;

char *datadir;
int useheight=1;
int usealtblend=0;
int fullscreen=0;
extern int psysblendmode;
int configged = 0;
int configuring = 0;
int halt = 1;
int initted = 0;
int running = 0;

int LoadConfig(void)
{
	long i=1000;
	HKEY hkey=0;

	char str[10];

	configged = 1;
	datadir = (char*)malloc(1024);

	sprintf(datadir, ".");
	if ( RegOpenKey(HKEY_LOCAL_MACHINE, "Software\\Gripp\\NUMB", &hkey) != ERROR_SUCCESS )
		if ( RegCreateKey(HKEY_LOCAL_MACHINE, "Software\\Gripp\\NUMB", &hkey) != ERROR_SUCCESS )
			return 0;
	i=1000;
	if ( RegQueryValue(hkey, "datadir", datadir, &i) != ERROR_SUCCESS )
		configged = 0;

	i=10;
	if ( RegQueryValue(hkey, "use_heightfield", str, &i) != ERROR_SUCCESS )
		configged = 0;
	else
		useheight = atoi(str);

	i=10;
	if ( RegQueryValue(hkey, "fullscreen", str, &i) != ERROR_SUCCESS )
		configged = 0;
	else
		fullscreen = atoi(str);

	if ( RegQueryValue(hkey, "use_alt_blend", str, &i) != ERROR_SUCCESS )
		configged = 0;
	else
		usealtblend = atoi(str);

	RegCloseKey(hkey);
	_chdir(datadir);
	return configged;
}


int SaveConfig(void)
{	long i=1000;
	HKEY hkey=0;

	char str[10];

	configged = 1;
	if ( RegOpenKey(HKEY_LOCAL_MACHINE, "Software\\Gripp\\NUMB", &hkey) == ERROR_SUCCESS )
	{	i = strlen(datadir);
		if ( RegSetValue(hkey, "datadir", REG_SZ, datadir, i) != ERROR_SUCCESS )
			MessageBox(NULL, "Couldn't save datadir", "Error", MB_ICONERROR);

		sprintf(str, "%d", useheight);
		i = strlen(str);
		RegSetValue(hkey, "use_heightfield", REG_SZ, str, i);
		
		sprintf(str, "%d", fullscreen);
		i = strlen(str);
		RegSetValue(hkey, "fullscreen", REG_SZ, str, i);
		
		sprintf(str, "%d", usealtblend);
		i = strlen(str);
		RegSetValue(hkey, "use_alt_blend", REG_SZ, str, i);

		RegCloseKey(hkey);
	}
	return 1;
}

int WINAPI WinMain(HINSTANCE hThisInst, HINSTANCE hPrevInst, LPSTR lpszArgs, int nWinMode)
{
	MSG msg;
	WNDCLASS wcl;
	int done;

	HWND hparent = NULL;

	SetEnvironmentVariable("FX_GLIDE_NOSPLASH", "1");

	if ( !LoadConfig() )
	{	MessageBox(NULL, "Couldn't read configuration settings from registry.", "Warning", MB_OK);
		configuring = 1;
	}

	if ( strncmp(lpszArgs, "/p", 2) == 0 )
	{//	MessageBox(hparent, "Can't run in preview mode.", "Warning", MB_OK);
			exit(0);
	}

	if ( strncmp(lpszArgs, "/c", 2) == 0 || configuring )
	{
		configuring = 1;
		/* create configuration dialog box */
		if ( (hDialog = CreateDialog(hInst, MAKEINTRESOURCE(IDD_CONFIGDIALOG), NULL, DlgProc )) == NULL )
		{	MessageBox(NULL, "Couldn't create dialog box", "Error", MB_ICONERROR);
			return 0;
		}
		while ( GetMessage(&msg, NULL, 0, 0) )
		{	TranslateMessage(&msg);	/* allow use of keyboard */
			DispatchMessage(&msg);	/* return control to Windows */
		}
		return 0;
	}

	/* set global hInst first-thing */
	hInst = hThisInst;

	if ( !configuring )
	{
		// Define a window class
		wcl.hInstance = hInst; // handle to this instance
		wcl.lpszClassName = lpszWinName; // window class name
		wcl.lpfnWndProc = WindowProc; // window function
		wcl.style = CS_VREDRAW | CS_HREDRAW | CS_SAVEBITS | CS_DBLCLKS; // style
		wcl.hIcon = LoadIcon(NULL, MAKEINTRESOURCE(IDI_APP)); // icon style
		wcl.hCursor = LoadCursor(NULL, IDC_ARROW); // cursor style
		wcl.lpszMenuName = NULL; // menu
		wcl.cbClsExtra = 0; // extra
		wcl.cbWndExtra = 0; // extra
		wcl.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);

		// Register the window class
		if ( !RegisterClass(&wcl) ) return 0;

		if ( fullscreen )
		{	modenum = 0;
			do
			{	done = !EnumDisplaySettings(NULL, modenum, &dmode );
				if ( dmode.dmBitsPerPel == 16 && dmode.dmPelsWidth == 640 && dmode.dmPelsHeight == 480 ) break;
				modenum++;
			} while ( !done );
			if ( done )	// didn't find mode
			{	MessageBox(NULL, "Didn't find good fullscreen mode", "Error", MB_ICONERROR);
				exit(-1);
			}

			dmode.dmFields = DM_PELSWIDTH | DM_PELSHEIGHT | DM_BITSPERPEL;

			switch ( ChangeDisplaySettings(&dmode, CDS_TEST) )
			{	case DISP_CHANGE_SUCCESSFUL: break;
				case DISP_CHANGE_FAILED:
					MessageBox(NULL, "DISP_CHANGE_FAILED", "Error", MB_ICONERROR);
					exit(-1);
				case DISP_CHANGE_BADMODE:
					MessageBox(NULL, "DISP_CHANGE_BADMODE", "Error", MB_ICONERROR);
					exit(-1);
			}
		}

		hWindow = CreateWindow
		(
			lpszWinName, // name of window class
			lpszAppName, // title
			WS_POPUP, // window style
			0, // X coordinate
			0, // Y coordinate
			winWidth, // width
			winHeight, // height
			NULL, // parent window
			NULL, // menu
			hInst, // handle of this instance of the program
			NULL // additional arguments
		);
	}// !configuring

	/* Create the message loop. */
	while ( !DoneWithMessageLoop )
	{	while ( PeekMessage(&msg, NULL, 0, 0, PM_REMOVE) )
		{	if ( msg.message == WM_QUIT )
			{	DoneWithMessageLoop = TRUE;
			}
			else
			{	TranslateMessage(&msg);	/* allow use of keyboard */
				DispatchMessage(&msg);	/* return control to Windows */
			}
		}
		if ( !DoneWithMessageLoop )
		{	running = 1;
			redraw();
		}
		else
		{	Sleep(50);
		}
	}
	return msg.wParam;
}


LRESULT CALLBACK WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{	static int mouseskip = 1;
	static FILE *msglog;

	if ( !msglog )
	{	msglog = fopen("msg.log", "w");
	}
	fprintf(msglog, "Message %X: wparam: %X lparam: %X\n", message, wParam, lParam);
	fflush(msglog);

	switch (message)
	{	case WM_CREATE:
			hWindow = hWnd;
			hDC = GetDC(hWindow);
			ShowCursor(FALSE);
			if ( fullscreen )
				ChangeDisplaySettings(&dmode, CDS_FULLSCREEN);
			else
				SetCursorPos(winWidth/2,winHeight/2);

			ShowWindow(hWindow, SW_SHOW);

			if ( !setupPixelFormat(hDC) )
			{	MessageBox(NULL, "Couldn't setup pixelformat", "Error:", MB_ICONERROR);
				DestroyWindow(hWindow);
			}
			if ( (hGLRC = wglCreateContext(hDC)) == NULL )
			{
				if ( fullscreen )
				{	ChangeDisplaySettings(NULL, CDS_FULLSCREEN);
				}
				MessageBox(NULL, "Couldn't create gl context\n"
										"Close display properties?\n"
										"Try unsetting fullscreen?", "Error:", MB_ICONERROR);
				DestroyWindow(hWindow);
			}
			if ( !wglMakeCurrent(hDC, hGLRC) )
			{	if ( fullscreen )
				{	ChangeDisplaySettings(NULL, CDS_FULLSCREEN);
				}
				MessageBox(NULL, "Couldn't wglMakeCurrent()", "Error:", MB_ICONERROR);
				DestroyWindow(hWindow);
			}
			init();
			initted = 1;
			mouseskip++;
			return 0;

		case WM_DESTROY:
		{
			ShowCursor(TRUE);
			if ( fullscreen )
			{	ChangeDisplaySettings(NULL, CDS_FULLSCREEN);
			}
			ShowWindow(hWnd, SW_HIDE);
			wglMakeCurrent(NULL,NULL);
			if ( hGLRC )
			{	wglDeleteContext(hGLRC);
			}
			timeEndPeriod(1);
			ReleaseDC(hWnd, hDC);
			fclose(msglog);
			PostQuitMessage(0);
			return 0;
		}

		case WM_PAINT:
		{	if ( initted )
			{	PAINTSTRUCT ps;
				BeginPaint(hWnd, &ps);
				EndPaint(hWnd, &ps);
			}
			return 0;
		}

		case WM_WINDOWPOSCHANGING:
			mouseskip++;
			break;

		case WM_MOUSEMOVE:
			if ( mouseskip )
			{	mouseskip--;
				break;
			}
			DestroyWindow(hWindow);
			return 0;

		case WM_ACTIVATE:
          if ( wParam != FALSE ) return 1;
		case WM_KEYDOWN:
		case WM_LBUTTONDOWN:
		case WM_MBUTTONDOWN:
		case WM_RBUTTONDOWN:
			if ( running ) DestroyWindow(hWindow);
			return 0;
		case WM_SETCURSOR: return 0;

		case WM_SYSCOMMAND:
			if ( wParam == SC_SCREENSAVE ) return 0;
		default:
			return DefWindowProc(hWnd, message, wParam, lParam);
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

void init(void)
{	GLdouble aspect, fovy, zNear, zFar;
	int i;
	char texname[1000];

	if ( initted ) return;

	initted = 1;
	glClearColor(0.0, 0.0, 0.0, 0.0);
	glClearDepth(1.0);
	glShadeModel(GL_SMOOTH);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	sprintf(texname, "%s\\particle.pcx", datadir);
	if ( !Load_Texture(texname, PARTICLE_TEXTURE) )
	{
		if ( fullscreen )
		{	ChangeDisplaySettings(NULL, CDS_FULLSCREEN);
		}
		MessageBox(NULL, "Error loading texture", "Error", MB_ICONERROR);
		DestroyWindow(hWindow);
	}
	sprintf(texname, "%s\\font.pcx", datadir);
	if ( !Font_LoadFont(texname, FONT_TEXNUM) )
	{	if ( fullscreen )
		{	ChangeDisplaySettings(NULL, CDS_FULLSCREEN);
		}
		MessageBox(NULL, "Error loading texture", "Error", MB_ICONERROR);
		DestroyWindow(hWindow);
	}
	sprintf(texname, "%s\\height.pcx", datadir);
	if ( !Load_Texture(texname, HEIGHTFIELD_TEXTURE) )
	{
		if ( fullscreen )
		{	ChangeDisplaySettings(NULL, CDS_FULLSCREEN);
		}
		MessageBox(NULL, "Error loading texture", "Error", MB_ICONERROR);
		DestroyWindow(hWindow);
	}

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	zNear = 1.01;
	zFar = 4096.0;
	aspect = (double)winWidth / (double) winHeight;
	fovy = 90.0;

	// set viewing projection
	glMatrixMode(GL_PROJECTION);
	gluPerspective(fovy, aspect, zNear, zFar);
	InitTimes();
	if ( timeBeginPeriod(1) == TIMERR_NOCANDO )
	{	MessageBox(hWindow, "Can't set timer resolution to 1 millisecond", "Error", MB_ICONERROR | MB_OK);
		DestroyWindow(hWindow);
	}
	UpdateTimes();

	i = Psys_New(GRAVFIELD);
	Psys_SetPosN(i, 0.0, 0.0, 100.0);
	psysblendmode = GL_ONE;
	if ( usealtblend ) psysblendmode = GL_ONE_MINUS_SRC_ALPHA;

	HField_Init(32, 32, 16);
	HField_StartEffect(WAVY1);

	viewangle[PITCH] = -10.0;
	viewpos[2] = 130.0;
	spinspeed = 10.0;

	halt = 0;
}

void redraw(void)
{	char str[100];

	UpdateTimes();
	UpdateViewPos();

	// clear color and depth buffers
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glDisable(GL_BLEND);

	// position viewer
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
		// this fixes axial issues between standard screen space
		// and my quake mapping world space
		glRotatef(-90, 1.0, 0.0, 0.0);
		glRotatef(90, 0.0, 0.0, 1.0);

		// rotate everything into the view vector
		glRotatef(viewangle[ROLL], 1.0F, 0.0F, 0.0F); // roll
		glRotatef(viewangle[PITCH], 0.0F, 1.0F, 0.0F); // up/down
		glRotatef(viewangle[YAW], 0.0F, 0.0F, 1.0F); // left-right

		// translate everything to view position
		glTranslatef(-viewpos[0], -viewpos[1], -viewpos[2]);

		if ( useheight )
		{	HField_Animate();
			HField_Draw();
		}

		Psys_AnimateAll();
		Psys_DrawAll();

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	Font_BlitString("NUMB, by Mike Ruete, for Operation 3DFX contest - Copyright 1998", 0, 470, FONT_TEXNUM);
	sprintf(str, "fps = %03.4f", (float)framenum / (lastframe - firstframe)*1000.0);
	Font_BlitString(str, 0, 0, FONT_TEXNUM);
//	Font_BlitString("Press ESCAPE to exit", 0, 10, FONT_TEXNUM);

	SwapBuffers(hDC);
}


void resize(void)
{
    /* set viewport to cover the window */
    glViewport(0, 0, winWidth, winHeight);
}

BOOL setupPixelFormat(HDC hDC)
{
	PIXELFORMATDESCRIPTOR pfd, *ppfd;
	int pixelFormat;

	ppfd = &pfd;
	ppfd->nSize = sizeof(PIXELFORMATDESCRIPTOR);
	ppfd->nVersion = 1;
	ppfd->dwFlags = PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	ppfd->dwLayerMask = PFD_MAIN_PLANE;
	ppfd->iPixelType = PFD_TYPE_RGBA;
	ppfd->cColorBits = 16;
	ppfd->cDepthBits = 16;
	ppfd->cAccumBits = 0;
	ppfd->cStencilBits = 0;

	pixelFormat = ChoosePixelFormat(hDC, &pfd);
	if ( pixelFormat == 0 )
	{	MessageBox(WindowFromDC(hDC), "ChoosePixelFormat failed.", "Error", MB_ICONERROR | MB_OK);
		exit(1);
	}

	if (pfd.dwFlags & PFD_NEED_PALETTE)
	{	MessageBox(NULL, "Needs palette", "Error", MB_OK);
		return FALSE;
	}

	if ( SetPixelFormat(hDC, pixelFormat, &pfd) != TRUE )
	{	MessageBox(WindowFromDC(hDC), "SetPixelFormat failed.", "Error", MB_ICONERROR | MB_OK);
		exit(1);
	}

	return TRUE;
}

void UpdateViewPos(void)
{	viewangle[YAW]+=framedelta * spinspeed;
	viewangle[ROLL] = 0.0;
	viewpos[0] = -120 * cos( viewangle[YAW] * M_PI / 180.0 );
	viewpos[1] = 120 * sin( viewangle[YAW] * M_PI / 180.0 );
}

void DrawStrip(vec3_t *edge1, vec3_t *edge2, int l)
{	int i=0;

	glDisable(GL_TEXTURE_2D);
	glDisable(GL_CULL_FACE);
	glBegin(GL_TRIANGLE_STRIP);
	for ( i=0; i<(l-1); i++ )
	{	glColor4f(0.0, 0.0, (float)i/(float)l, 1.0);
		//glColor4f(1.0, 1.0, 1.0, 1.0);
		glVertex3fv( edge1[i] );

		glColor4f(0.0, 0.0, (float)i+1/(float)l, 1.0);

		glVertex3fv( edge2[i] );
	}
	glEnd();
}


void InitTimes(void)
{
	lasttime = timeGetTime();
	framedelta = 0.0;
	firstframe = lasttime;
	lastframe = firstframe;
}

void UpdateTimes(void)
{	int thistime;

	if ( framenum>10 )
	{	framenum = 0;
		firstframe = lastframe;
	}

	thistime = timeGetTime();
	framedelta = (float)(thistime - lasttime)/1000.0;
	
	lasttime = thistime;
	lastframe = thistime;
	framenum++;
}


BOOL CALLBACK DlgProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{	static FILE *msglog;

	if ( !msglog )
	{	msglog = fopen("dlgmsg.log", "w");
	}
/*	fprintf(msglog, "Message %X: wparam: %X lparam: %X\n", uMsg, wParam, lParam);
	fflush(msglog);
*/
	switch (uMsg)
	{
		case WM_INITDIALOG:
			hDialog = hwndDlg;
			//MessageBox(NULL, datadir, "Setting datadir:", MB_OK);
			SetDlgItemText(hDialog, IDC_DATADIR, datadir);
			SendDlgItemMessage(hDialog, IDC_USEHEIGHTFIELD, BM_SETCHECK, useheight, 0);
			SendDlgItemMessage(hDialog, IDC_USEALTBLEND, BM_SETCHECK, usealtblend, 0);
			SendDlgItemMessage(hDialog, IDC_FULLSCREEN, BM_SETCHECK, fullscreen, 0);
			return 1;

		case WM_CLOSE:
		case WM_DESTROY:
//			fclose(msglog);
			EndDialog(hwndDlg, 1);
			PostQuitMessage(0);
			return 0;
		
		case WM_COMMAND: 
			switch (LOWORD(wParam))
			{	case WM_DESTROY:
					DestroyWindow(hDialog);
					return 0;
				case IDOK:
					GetDlgItemText(hDialog, IDC_DATADIR, datadir, 1000);
					useheight = SendDlgItemMessage(hDialog, IDC_USEHEIGHTFIELD, BM_GETCHECK, 0, 0);
					usealtblend = SendDlgItemMessage(hDialog, IDC_USEALTBLEND, BM_GETCHECK, 0, 0);
					fullscreen = SendDlgItemMessage(hDialog, IDC_FULLSCREEN, BM_GETCHECK, 0, 0);
					//MessageBox(NULL, datadir, "Datadir:", MB_OK);
					SaveConfig();
					DestroyWindow(hDialog);
					return 0;

				default: return 1;
			}	
			break;
		default:
			return 0;
	}
	return 1;
}
